/*
 * ObservedValuesContainer.hpp
 *
 *  Created on: Oct 29, 2015
 *      Author: heber
 */


#ifndef OBSERVEDVALUESCONTAINER_HPP_
#define OBSERVEDVALUESCONTAINER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ObservedValue_types.hpp"

#include <deque>
#include <map>
#include <string>

#include <boost/function.hpp>

class QtObservedInstanceBoard;

/** This class contains ObservedValues of the class \b T each instance identified
 * by the id type \b id.
 *
 * All the reference counting is done inside this container.
 */
template <class T, typename id>
class ObservedValuesContainer
{
public:

  //!> typedef for callback functions to be used on last SubjectKilled()
  typedef boost::function<void (const id _id)> onDestroy_t;

  /** Cstor of class ObservedValuesContainer.
   *
   * \param _name name used in debugging and prints
   * \param _board ref to InstanceBoard
   * \param _onDestroy function to call when last subjectKilled() was received and
   *        ObservedValues are destroyed
   */
  ObservedValuesContainer(
      const std::string _name,
      QtObservedInstanceBoard &_board,
      const onDestroy_t _onDestroy);

  /** Destor of class ObservedValuesContainer.
   *
   */
  ~ObservedValuesContainer();

  /** Delivers the set of Observed value for the instance identified by \a _id.
   *
   * \param _id identifier of the instance
   * \return shared ptr to observed instance.
   */
  typename T::ptr get(const id _id);

  /** Used by QtObserved.. instance to note that signOn() has been called.
   *
   * \param _id identifier of the instance who called signOn()
   */
  void markObservedValuesAsConnected(const id _id);

  /** Used by QtObserved.. instance to note that signOff() has been called.
   *
   * \param _id identifier of the instance who called signOff()
   */
  void markObservedValuesAsDisconnected(const id _id);

  /** Prepares removeal a vector of observed values of an instance  identified by \a _id.
   *
   * \param _id identifier of instance
   */
  void removeObservedValues(const id _id);

  /** Erases a vector of observed values of an instance  identified by \a _id.
   *
   * \param _id identifier of instance
   */
  void eraseObservedValues(const id _id);

private:

  typedef std::map<id, typename T::ptr > CountedObservedValues_t;
  //!> internal vector of observed values
  CountedObservedValues_t ObservedValues;

  //!> typedef for the set with ids to be erase
  typedef std::set<id> MarkedSet_t;

  //!> marks ids marked for erase (i.e. all subjectKilled() received)
  MarkedSet_t MarkedForErase;

  //!> marks ids marked as connected (i.e. for later subjectKilled() received)
  MarkedSet_t MarkedForConnected;

  //!> name used in describing the instance type
  const std::string NameOfType;

  //!> reference to InstanceBoard for callbacks on subjectKilled()
  QtObservedInstanceBoard &board;

  //!> callback function when ObservedValues need to be destroyed
  const onDestroy_t onDestroy;

  //!> internal mutex to have atomic access to control maps
  mutable boost::recursive_mutex atomic_mutex;

private:
  /** Internal function to check whether an Observed instance identified by
   * \a _id is still signOn() to its associated World instance.
   *
   * \param _id identifier of instance
   * \return true - no more signOn()s, false - else
   */
  bool checkMarkedForConnected(const id _id) const;

  /** Internal function to check whether the vector of ObservedValue's
   *  identified by \a _id has been marked for erase.
   *
   *  Marked for erase means that it has received all subjectKilled()
   *  (the container not the values themselves).
   *
   * \param _id identifier of instance
   * \return true - marked for erase, false - else
   */
  bool checkMarkedForErase(const id _id) const;

  /** Combines all of the above checks with call to removeObservedValues() on true.
   *
   * \param _id identifier of instance
   */
  void checkRemoval(const id _id);

private:
  //!> QtObservedInstanceBoard may access anything
  friend class QtObservedInstanceBoard;

  /** Inserts a new ObservedValue vector into the container.
   *
   * \param _id identifier of instance associated with observed values
   * \param _obsvalues vector of observed values of instance
   */
  void insert(const id _id, const typename T::ptr &_obsvalues);

  /** Use to change the identifier associated with a vector of observed values.
   *
   * \param _oldid old identifier
   * \param _newid new identifier
   * \return true - change successful, false - else
   */
  bool changeIdentifier(const id _oldid, const id _newid);

  /** Checks whether a vector of observed values of an instance  identified by \a _id
   * is present.
   *
   * \param _id identifier of instance
   * \return true - present, false - else
   */
  bool isPresent(const id _id) const;
};

#endif /* OBSERVEDVALUESCONTAINER_HPP_ */
