/*
 * MoleculeDescriptor.hpp
 *
 *  Created on: Feb 5, 2010
 *      Author: crueger
 */

#ifndef MOLECULEDESCRIPTOR_HPP_
#define MOLECULEDESCRIPTOR_HPP_


#include <vector>
#include <map>
#include <boost/shared_ptr.hpp>
#include "World.hpp"

class World;
class molecule;

// internal implementation, allows assignment, copying etc
class MoleculeDescriptor_impl;

/**
 * An MoleculeDescriptor describes a Set of Molecules from the World. Can be used for any method that needs to work on
 * a specific set of Molecules.
 *
 * This Class is implemented using the PIMPL-Idion, i.e. this class only contains an abstract structure
 * that forwards any request to a wrapped pointer-to-implementation. This way operators and calculations
 * on Descriptors are possible.
 *
 * Concrete Implementation Objects can be shared between multiple Wrappers, so make sure that
 * any Implementation remainst constant during lifetime.
 */
class MoleculeDescriptor {
  // close coupling to the world to allow access
  friend molecule* World::getMolecule(MoleculeDescriptor descriptor);
  friend std::vector<molecule*> World::getAllMolecules(MoleculeDescriptor descriptor);
  template <class,class,class> friend class SelectiveIterator;

  friend MoleculeDescriptor operator&&(const MoleculeDescriptor &lhs, const MoleculeDescriptor &rhs);
  friend MoleculeDescriptor operator||(const MoleculeDescriptor &lhs, const MoleculeDescriptor &rhs);
  friend MoleculeDescriptor operator!(const MoleculeDescriptor &arg);

public:
  typedef MoleculeDescriptor_impl impl_t;
  typedef boost::shared_ptr<impl_t> impl_ptr; //!< Allow easy changes of the pointer-to-implementation type

  MoleculeDescriptor(impl_ptr);

  /**
   * Copy constructor.
   * Takes the Implementation from the copied object and sets it's own pointer to link there.
   * This way the actuall implementation object is shared between copy and original
   */
  MoleculeDescriptor(const MoleculeDescriptor&);
  ~MoleculeDescriptor();

  /**
   * Assignment Operator.
   *
   * Implemented by setting the pointer to the new Implementation.
   */
  MoleculeDescriptor &operator=(MoleculeDescriptor &);

protected:
  /**
   * forward Method to implementation
   */
  molecule* find();

  /**
   * forward Method to implementation
   */
  std::vector<molecule*> findAll();

  /**
   * Return the implementation this Wrapper currently points to.
   * Used for copying, assignment and in Iterators over subsets of the World.
   */
  impl_ptr get_impl() const;

private:
  impl_ptr impl;
};

/**
 * produce an Moleculedescriptor that at the point of construction contains an implementation that matches all Molecules
 */
MoleculeDescriptor AllMolecules();

/**
 * produce an Moleculedescriptor that at the point of construction contains an implementation that matches no Molecules
 */
MoleculeDescriptor NoMolecules();

/**
 * Set Intersection for two Moleculedescriptors. The resulting Moleculedescriptor will only match an Molecule if both
 * given Moleculedescriptors also match. Uses short circuit inside, so the second predicate wont be called
 * when the first one failed.
 */
MoleculeDescriptor operator&&(const MoleculeDescriptor &lhs, const MoleculeDescriptor &rhs);

/**
 * Set Union for two MoleculeDescriptors. The resulting MoleculeDescriptor will match an Molecule if at least one of
 * the two given MoleculeDescriptors does match. Used short circuit inside, so the second predicate wont
 * be called when the first one failed.
 */
MoleculeDescriptor operator||(const MoleculeDescriptor &lhs, const MoleculeDescriptor &rhs);

/**
 * Set inversion for an MoleculeDescriptor. Matches an Molecule if the given MoleculeDescriptor did not match.
 */
MoleculeDescriptor operator!(const MoleculeDescriptor &arg);

#endif /* MOLECULEDESCRIPTOR_HPP_ */
