/*
 * AtomDescriptor.hpp
 *
 *  Created on: Feb 5, 2010
 *      Author: crueger
 */

#ifndef ATOMDESCRIPTOR_HPP_
#define ATOMDESCRIPTOR_HPP_


#include <vector>
#include <map>
#include <boost/shared_ptr.hpp>
#include "World.hpp"

class World;
class atom;

// internal implementation, allows assignment, copying etc
class AtomDescripter_impl;

/**
 * An AtomDescriptor describes a Set of Atoms from the World. Can be used for any method that needs to work on
 * a specific set of Atoms.
 *
 * This Class is implemented using the PIMPL-Idion, i.e. this class only contains an abstract structure
 * that forwards any request to a wrapped pointer-to-implementation. This way operators and calculations
 * on Descriptors are possible.
 *
 * Concrete Implementation Objects can be shared between multiple Wrappers, so make sure that
 * any Implementation remainst constant during lifetime.
 */
class AtomDescriptor {
  // close coupling to the world to allow access
  friend atom* World::getAtom(AtomDescriptor descriptor);
  friend std::vector<atom*> World::getAllAtoms(AtomDescriptor descriptor);
  template <class,class,class> friend class SelectiveIterator;

  friend AtomDescriptor operator&&(const AtomDescriptor &lhs, const AtomDescriptor &rhs);
  friend AtomDescriptor operator||(const AtomDescriptor &lhs, const AtomDescriptor &rhs);
  friend AtomDescriptor operator!(const AtomDescriptor &arg);

public:
  typedef boost::shared_ptr<AtomDescriptor_impl> impl_ptr; //!< Allow easy changes of the pointer-to-implementation type

  AtomDescriptor(impl_ptr);

  /**
   * Copy constructor.
   * Takes the Implementation from the copied object and sets it's own pointer to link there.
   * This way the actuall implementation object is shared between copy and original
   */
  AtomDescriptor(const AtomDescriptor&);
  ~AtomDescriptor();

  /**
   * Assignment Operator.
   *
   * Implemented by setting the pointer to the new Implementation.
   */
  AtomDescriptor &operator=(AtomDescriptor &);

protected:
  /**
   * forward Method to implementation
   */
  atom* find();

  /**
   * forward Method to implementation
   */
  std::vector<atom*> findAll();

  /**
   * Return the implementation this Wrapper currently points to.
   * Used for copying, assignment and in Iterators over subsets of the World.
   */
  impl_ptr get_impl() const;

private:
  impl_ptr impl;
};

/**
 * produce an Atomdescriptor that at the point of construction contains an implementation that matches all Atoms
 */
AtomDescriptor AllAtoms();

/**
 * produce an Atomdescriptor that at the point of construction contains an implementation that matches no Atoms
 */
AtomDescriptor NoAtoms();

/**
 * Set Intersection for two Atomdescriptors. The resulting Atomdescriptor will only match an Atom if both
 * given Atomdescriptors also match. Uses short circuit inside, so the second predicate wont be called
 * when the first one failed.
 */
AtomDescriptor operator&&(const AtomDescriptor &lhs, const AtomDescriptor &rhs);

/**
 * Set Union for two AtomDescriptors. The resulting AtomDescriptor will match an Atom if at least one of
 * the two given AtomDescriptors does match. Used short circuit inside, so the second predicate wont
 * be called when the first one failed.
 */
AtomDescriptor operator||(const AtomDescriptor &lhs, const AtomDescriptor &rhs);

/**
 * Set inversion for an AtomDescriptor. Matches an Atom if the given AtomDescriptor did not match.
 */
AtomDescriptor operator!(const AtomDescriptor &arg);

#endif /* ATOMDESCRIPTOR_HPP_ */
