/*
 * atom_bondedparticle.cpp
 *
 *  Created on: Oct 19, 2009
 *      Author: heber
 */

#include "atom.hpp"
#include "atom_bondedparticle.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "lists.hpp"
#include "verbose.hpp"

/** Constructor of class BondedParticle.
 */
BondedParticle::BondedParticle()
{
};

/** Destructor of class BondedParticle.
 */
BondedParticle::~BondedParticle()
{
  BondList::const_iterator Runner;
  while (!ListOfBonds.empty()) {
    Runner = ListOfBonds.begin();
    removewithoutcheck(*Runner);
  }
};

/** Outputs the current atom::AdaptiveOrder and atom::MaxOrder to \a *file.
 * \param *file output stream
 */
void BondedParticle::OutputOrder(ofstream *file) const
{
  *file << nr << "\t" << (int)AdaptiveOrder << "\t" << (int)MaxOrder << endl;
  //cout << Verbose(2) << "Storing: " << nr << "\t" << (int)AdaptiveOrder << "\t" << (int)MaxOrder << "." << endl;
};

/** Prints all bonds of this atom with total degree.
 * \param *out stream to output to
 * \return true - \a *out present, false - \a *out is NULL
 */
bool BondedParticle::OutputBondOfAtom(ofstream *out) const
{
  if (out != NULL) {
    *out << Verbose(4) << "Atom " << Name << "/" << nr << " with " << ListOfBonds.size() << " bonds: ";
    int TotalDegree = 0;
    for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); ++Runner) {
      *out << **Runner << "\t";
      TotalDegree += (*Runner)->BondDegree;
    }
    *out << " -- TotalDegree: " << TotalDegree << endl;
    return true;
  } else
    return false;
};

/** Output of atom::nr along with all bond partners.
 * \param *AdjacencyFile output stream
 */
void BondedParticle::OutputAdjacency(ofstream *AdjacencyFile) const
{
  *AdjacencyFile << nr << "\t";
  for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); (++Runner))
    *AdjacencyFile << (*Runner)->GetOtherAtom(this)->nr << "\t";
  *AdjacencyFile << endl;
};

/** Puts a given bond into atom::ListOfBonds.
 * \param *Binder bond to insert
 */
bool BondedParticle::RegisterBond(bond *Binder)
{
  bool status = false;
  if (Binder != NULL) {
    if (Binder->Contains(this)) {
      ListOfBonds.push_back(Binder);
      status = true;
    } else {
      cout << Verbose(1) << "ERROR: " << *Binder << " does not contain " << *this << "." << endl;
    }
  } else {
    cout << Verbose(1) << "ERROR: Binder is " << Binder << "." << endl;
  }
  return status;
};

/** Removes a given bond from atom::ListOfBonds.
 * \param *Binder bond to remove
 */
bool BondedParticle::UnregisterBond(bond *Binder)
{
  bool status = false;
  if (Binder != NULL) {
    if (Binder->Contains(this)) {
      ListOfBonds.remove(Binder);
      status = true;
    } else {
      cout << Verbose(1) << "ERROR: " << *Binder << " does not contain " << *this << "." << endl;
    }
  } else {
    cout << Verbose(1) << "ERROR: Binder is " << Binder << "." << endl;
  }
  return status;
};

/** Removes all bonds from atom::ListOfBonds.
 * \note Does not do any memory de-allocation.
 */
void BondedParticle::UnregisterAllBond()
{
  ListOfBonds.clear();
};

/** Corrects the bond degree by one at most if necessary.
 * \param *out output stream for debugging
 */
int BondedParticle::CorrectBondDegree(ofstream *out)
{
  int NoBonds = 0;
  int OtherNoBonds = 0;
  int FalseBondDegree = 0;
  atom *OtherWalker = NULL;
  bond *CandidateBond = NULL;

  //*out << Verbose(3) << "Walker " << *this << ": " << (int)this->type->NoValenceOrbitals << " > " << NoBonds << "?" << endl;
  NoBonds = CountBonds();
  if ((int)(type->NoValenceOrbitals) > NoBonds) { // we have a mismatch, check all bonding partners for mismatch
    for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); (++Runner)) {
      OtherWalker = (*Runner)->GetOtherAtom(this);
      OtherNoBonds = OtherWalker->CountBonds();
      //*out << Verbose(3) << "OtherWalker " << *OtherWalker << ": " << (int)OtherWalker->type->NoValenceOrbitals << " > " << NoBonds << "?" << endl;
      if ((int)(OtherWalker->type->NoValenceOrbitals) > NoBonds) { // check if possible candidate
        if ((CandidateBond == NULL) || (ListOfBonds.size() > OtherWalker->ListOfBonds.size())) { // pick the one with fewer number of bonds first
          CandidateBond = (*Runner);
          //*out << Verbose(3) << "New candidate is " << *CandidateBond << "." << endl;
        }
      }
    }
    if ((CandidateBond != NULL)) {
      CandidateBond->BondDegree++;
      *out << Verbose(2) << "Increased bond degree for bond " << *CandidateBond << "." << endl;
    } else {
      //*out << Verbose(2) << "Could not find correct degree for atom " << *this << "." << endl;
      FalseBondDegree++;
    }
  }
  return FalseBondDegree;
};

/** Checks whether there is a bond between \a this atom and the given \a *BondPartner.
 * \param *BondPartner atom to check for
 * \return true - bond exists, false - bond does not exist
 */
bool BondedParticle::IsBondedTo(BondedParticle * const BondPartner)
{
  bool status = false;

  for (BondList::iterator runner = ListOfBonds.begin(); runner != ListOfBonds.end(); runner++) {
    status = status || ((*runner)->Contains(BondPartner));
  }
  return status;
};

