/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * PairPotential_AngleUnitTest.cpp
 *
 *  Created on: Oct 16, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "PairPotential_AngleUnitTest.hpp"

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/FunctionArgument.hpp"
#include "Potentials/Specifics/PairPotential_Angle.hpp"
#include "Potentials/helpers.hpp"

using namespace boost::assign;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( PairPotential_AngleTest );

const double spring_constant = .5;

void PairPotential_AngleTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // create three distances to have the desired angle
  typedef std::vector<double> angles_t;
  angles_t angles;
  angles += -0.5, -.4, -.3, -.2, -.1, 0., .1, .2, .3, .4, .5;
  FunctionModel::arguments_t tempvector(3);
  tempvector[0] = argument_t( std::make_pair(0,1), std::make_pair(0,1), 1.);
  tempvector[1] = argument_t( std::make_pair(1,2), std::make_pair(1,1), 0.);
  tempvector[2] = argument_t( std::make_pair(0,2), std::make_pair(0,1), 1.);
  for (angles_t::const_iterator iter = angles.begin(); iter != angles.end(); ++iter) {
    tempvector[1].distance = sqrt(-(2.*(*iter)-2.));//1.+1.-x/(2.*1.*1.)
    input += tempvector;
  }
  output +=
      spring_constant*0.25,
      spring_constant*0.16,
      spring_constant*0.09,
      spring_constant*0.04,
      spring_constant*0.01,
      spring_constant*0.,
      spring_constant*0.01,
      spring_constant*0.04,
      spring_constant*0.09,
      spring_constant*0.16,
      spring_constant*0.25;

  CPPUNIT_ASSERT_EQUAL( input.size(), output.size() );
}


void PairPotential_AngleTest::tearDown()
{
}

/** UnitTest for operator()
 */
void PairPotential_AngleTest::operatorTest()
{
  PairPotential_Angle::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Angle::ParticleType_t>
        (0)(1)(1)
      ;
  PairPotential_Angle angle(types, spring_constant,0.);
  for (size_t index = 0; index < input.size(); ++index) {
    const PairPotential_Angle::results_t result =
        angle( input[index] );
    CPPUNIT_ASSERT(
        Helpers::isEqual(
            output[index],
            result[0]
        )
    );
  }
}

/** UnitTest for derivative()
 */
void PairPotential_AngleTest::derivativeTest()
{
  PairPotential_Angle::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Angle::ParticleType_t>
        (0)(1)(1)
      ;
  PairPotential_Angle angle(types, spring_constant,0.);
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          angle.derivative(
              input[5]
          )[0],
          10.
      )
  );
}


/** UnitTest for parameter_derivative()
 */
void PairPotential_AngleTest::parameter_derivativeTest()
{
  PairPotential_Angle::ParticleTypes_t types =
      boost::assign::list_of<PairPotential_Angle::ParticleType_t>
        (0)(1)(1)
      ;
  PairPotential_Angle angle(types, spring_constant,0.);
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          angle.parameter_derivative(
              input[5],
              0
          )[0],
          10.
      )
  );
  CPPUNIT_ASSERT(
      Helpers::isEqual(
          0.,
          angle.parameter_derivative(
              input[5],
              1
          )[0],
          10.
      )
  );
}
