/*
 * element.hpp
 *
 *  Created on: Aug 3, 2009
 *      Author: heber
 */

#ifndef ELEMENT_HPP_
#define ELEMENT_HPP_

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <string>

#include "types.hpp"

#include <boost/version.hpp>
#if BOOST_VERSION > 106300
#include <boost/serialization/array_wrapper.hpp>
#endif
#include "boost/serialization/array.hpp"
#include "boost/serialization/string.hpp"

class periodentafel;
class IonTest;

/********************************************** declarations *******************************/

/** Chemical element.
 * Class incorporates data for a certain chemical element to be referenced from atom class.
 */
class element {
  friend class periodentafel;
  friend class IonTest;
  public:
    element();
    element(const element&);
    virtual ~element();

    element &operator=(const element&);

    // accessor functions
    double getMass() const;
    const unsigned char *getColor() const;
    double getCovalentRadius() const;
    double getElectronegativity() const;
    double getVanDerWaalsRadius() const;
    atomicNumber_t getAtomicNumber() const;
    virtual double getCharge() const
	{ return 0.; }
    virtual double getValence() const;
    virtual int getNoValenceOrbitals() const;
    double getHBondDistance(const size_t i) const;
    double getHBondAngle(const size_t i) const;

    const std::string &getSymbol() const;
    void setSymbol(const std::string &temp);

    const std::string &getName() const;
    void setName(const std::string &temp);

    bool operator==(const element &other) const;

    bool operator!=(const element &other) const {
      return !(*this == other);
    }

  private:
    friend class boost::serialization::access;
    // serialization
    template<class Archive>
    void serialize(Archive & ar, const unsigned int version)
    {
      ar & mass;
      ar & CovalentRadius;
      ar & Electronegativity;
      ar & VanDerWaalsRadius;
      ar & Z;
      ar & period;
      ar & group;
      ar & block;
      ar & Valence;
      ar & NoValenceOrbitals;
      ar & boost::serialization::make_array<double>(HBondDistance, 3);
      ar & boost::serialization::make_array<double>(HBondAngle, 3);
      ar & boost::serialization::make_array<unsigned char>(color, 3);
      ar & name;
      ar & symbol;
    }

  protected:

    double mass;    //!< mass in g/mol
    double CovalentRadius;  //!< covalent radius
    double Electronegativity; //!< electronegativity in Pauling units
    double VanDerWaalsRadius;  //!< van-der-Waals radius
    atomicNumber_t Z;          //!< atomic number
    std::string period;    //!< period: n quantum number
    std::string group;    //!< group: l quantum number
    std::string block;    //!< block: l quantum number
    double Valence;   //!< number of valence electrons for this element
    int NoValenceOrbitals;  //!< number of valence orbitals, used for determining bond degree in molecule::CreateConnectmatrix()
    double HBondDistance[3]; //!< distance in Angstrom of this element to hydrogen  (for single, double and triple bonds)
    double HBondAngle[3];     //!< typical angle for one, two, three bonded hydrogen (in degrees)
    unsigned char color[3];   //!< typical color for this element (from Jmol)

    std::string name;  //!< atom name, i.e. "Hydrogen"
    std::string symbol; //!< short form of the atom, i.e. "H"
};

std::ostream &operator<<(std::ostream&,const element&);

#endif /* ELEMENT_HPP_ */
