/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Dialog.cpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Dialog.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include "Parameters/ParameterExceptions.hpp"

class Atom;
class element;
class RealSpaceMatrix;
class molecule;
class Vector;

using namespace std;

Dialog::Dialog()
{
}

Dialog::~Dialog()
{
  list<Query*>::iterator iter;
  for(iter=queries.begin();iter!=queries.end();iter++){
    delete (*iter);
  }
}

void Dialog::registerQuery(Query *query){
  queries.push_back(query);
}

bool Dialog::display(){
  if(checkAll()){
    setAll();
    return true;
  }
  else{
    return false;
  }
}

bool Dialog::checkAll(){
  list<Query*>::iterator iter;
  bool retval = true;
  for(iter=queries.begin(); iter!=queries.end(); iter++){
    try {
      retval &= (*iter)->handle();
    } catch (ParameterException &e) {
      if( const std::string *name=boost::get_error_info<ParameterName>(e) )
        ELOG(1, "The following parameter value is not valid: " << *name << ".");
      retval = false;
      break;
    }
    // if any query fails (is canceled), we can end the handling process
    if(!retval) {
      ELOG(1, "The following query failed: " << (**iter).getTitle() << ".");
      break;
    }
  }
  return retval;
}

void Dialog::setAll(){
  list<Query*>::iterator iter;
  for(iter=queries.begin(); iter!=queries.end(); iter++) {
    try {
      (*iter)->setResult();
    } catch (ParameterException &e) {
      if( const std::string *name=boost::get_error_info<ParameterName>(e) )
        ELOG(1, "The following parameter value is not valid: " << *name << ".");
      break;
    }
  }
}

bool Dialog::hasQueries(){
  return queries.size();
}

/*template <> void Dialog::query<Dialog::EmptyType>(Parameter<Dialog::EmptyType> &param, const char *token, std::string description)
{
  queryEmpty(param, token, description);
}*/

template <> void Dialog::query<bool>(Parameter<bool> &param, const char *token, std::string description)
{
  queryBoolean(param, token, description);
}

template <> void Dialog::query<int>(Parameter<int> &param, const char *token, std::string description)
{
  queryInt(param, token, description);
}

template <> void Dialog::query< std::vector<int> >(Parameter<std::vector<int> > &param, const char *token, std::string description)
{
  queryInts(param, token, description);
}

template <> void Dialog::query<unsigned int>(Parameter<unsigned int> &param, const char *token, std::string description)
{
  queryUnsignedInt(param, token, description);
}

template <> void Dialog::query< std::vector<unsigned int> >(Parameter<std::vector<unsigned int> > &param, const char *token, std::string description)
{
  queryUnsignedInts(param, token, description);
}

template <> void Dialog::query<double>(Parameter<double> &param, const char *token, std::string description)
{
  queryDouble(param, token, description);
}

template <> void Dialog::query< std::vector<double> >(Parameter<std::vector<double> > &param, const char *token, std::string description)
{
  queryDoubles(param, token, description);
}

template <> void Dialog::query<std::string>(Parameter<std::string> &param, const char *token, std::string description)
{
  queryString(param, token, description);
}

template <> void Dialog::query< std::vector<std::string> >(Parameter<std::vector<std::string> > &param, const char *token, std::string description)
{
  queryStrings(param, token, description);
}

template <> void Dialog::query<const atom *>(Parameter<const atom *> &param, const char *token, std::string description)
{
  queryAtom(param, token, description);
}

template <> void Dialog::query< std::vector<const atom *> >(Parameter<std::vector<const atom *> > &param, const char *token, std::string description)
{
  queryAtoms(param, token, description);
}

template <> void Dialog::query<const molecule *>(Parameter<const molecule *> &param, const char *token, std::string description)
{
  queryMolecule(param, token, description);
}

template <> void Dialog::query< std::vector<const molecule *> >(Parameter<std::vector<const molecule *> > &param, const char *token, std::string description)
{
  queryMolecules(param, token, description);
}

template <> void Dialog::query<Vector>(Parameter<Vector> &param, const char *token, std::string description)
{
  queryVector(param, token, description);
}

template <> void Dialog::query< std::vector<Vector> >(Parameter<std::vector<Vector> > &param, const char *token, std::string description)
{
  queryVectors(param, token, description);
}

template <> void Dialog::query<RealSpaceMatrix>(Parameter<RealSpaceMatrix> &param, const char *token, std::string description)
{
  queryRealSpaceMatrix(param, token, description);
}

template <> void Dialog::query<const element *>(Parameter<const element *> &param, const char *token, std::string description)
{
  queryElement(param, token, description);
}

template <> void Dialog::query< std::vector<const element *> >(Parameter<std::vector<const element *> > &param, const char *token, std::string description)
{
  queryElements(param, token, description);
}

template <> void Dialog::query< boost::filesystem::path >(Parameter<boost::filesystem::path> &param, const char *token, std::string description)
{
  queryFile(param, token, description);
}

template <> void Dialog::query< std::vector<boost::filesystem::path> >(Parameter<std::vector< boost::filesystem::path> > &param, const char *token, std::string description)
{
  queryFiles(param, token, description);
}

template <> void Dialog::query< RandomNumberDistribution_Parameters >(Parameter<RandomNumberDistribution_Parameters> &param, const char *token, std::string description)
{
  queryRandomNumberDistribution_Parameters(param, token, description);
}

/************************** Query Infrastructure ************************/
/*       ---> shifted to folder Query                                   */
/************************************************************************/
