/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MatrixVector_ops.cpp
 *
 *  Created on: Nov 18, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <gsl/gsl_blas.h>
#include <gsl/gsl_vector.h>

#include "CodePatterns/Assert.hpp"
#include "MatrixContent.hpp"
#include "MatrixVector_ops.hpp"
#include "Vector.hpp"
#include "VectorContent.hpp"

/** Vector-Matrix-product.
 * @param vec vector
 * @param mat matrix
 * @return \f$(\mathrm{vec} \cdot \mathrm{mat})\f$
 */
VectorContent const operator*(const VectorContent& vec, const MatrixContent& mat)
{
  ASSERT( mat.columns == vec.dimension,
      "operator*() - column dimension of matrix ("+toString(mat.columns)
      +") and dimension of this vector ("+toString(vec.dimension)+") not equal!");
  VectorContent result(mat.rows);
  gsl_blas_dgemv( CblasNoTrans, 1.0, mat.content, vec.content, 0.0, result.content);
  return result;
}

/** Matrix-Vector-product.
 * @param mat matrix
 * @param vec vector
 * @return \f$(\mathrm{mat} \cdot \mathrm{vec})\f$
 */
VectorContent const operator*(const MatrixContent& mat, const VectorContent& vec)
{
  ASSERT( mat.columns == vec.dimension,
      "operator*() - column dimension of this matrix ("+toString(mat.columns)
      +") and dimension of vector ("+toString(vec.dimension)+") not equal!");
  VectorContent result(mat.rows);
  gsl_blas_dgemv( CblasNoTrans, 1.0, mat.content, vec.content, 0.0, result.content);
  return result;
}

/** Matrix-Vector-product.
 * @param mat matrix
 * @param vec vector
 * @return \f$(\mathrm{mat} \cdot \mathrm{vec})\f$
 */
Vector const operator*(const MatrixContent &mat, const Vector &vec)
{
  VectorContent res = mat*(vec.content);
  return Vector (res);
}

/** Matrix-Vector-product.
 * @param vec vector
 * @param mat matrix
 * @return \f$(\mathrm{vec} \cdot \mathrm{mat})\f$
 */
Vector const operator*(const Vector &vec, const MatrixContent &mat)
{
  VectorContent res = (vec.content)*mat;
  return Vector(res);
}

