/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * EnrollInPoolOperation.cpp
 *
 *  Created on: Feb 26, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "EnrollInPoolOperation.hpp"

#include <boost/bind.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "Jobs/FragmentJob.hpp"
#include "WorkerAddress.hpp"
#include "WorkerChoices.hpp"

/// Handle completion of a connect operation.
void EnrollInPoolOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the list
    // of jobs. The connection::async_read() function will automatically
    // decode the data that is read from the underlying socket.
    connection_.async_write(address,
      boost::bind(&EnrollInPoolOperation::handle_SendChoice, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&EnrollInPoolOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
  }
}


/// Handle sending choice
void EnrollInPoolOperation::handle_SendChoice(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the list
    // of jobs. The connection::async_read() function will automatically
    // decode the data that is read from the underlying socket.
    LOG(1, "INFO: Enrolling in pool with " << address << " ...");
    enum WorkerChoices choice = EnrollInPool;
    connection_.async_write(choice,
      boost::bind(&EnrollInPoolOperation::handle_ReceiveFlag, this,
      boost::asio::placeholders::error));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
  }
}


/// Handle receiving flag
void EnrollInPoolOperation::handle_ReceiveFlag(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the list
    // of jobs. The connection::async_read() function will automatically
    // decode the data that is read from the underlying socket.
    connection_.async_read(EnrollmentFlag,
      boost::bind(&EnrollInPoolOperation::handle_FinishOperation, this,
      boost::asio::placeholders::error));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
  }
}

/// Handle received flag
void EnrollInPoolOperation::handle_FinishOperation(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);

  if (EnrollmentFlag)
    LOG(2, "DEBUG: Enrollment successful.");
  else
    LOG(2, "DEBUG: Enrollment failed.");

  AsyncOperation::handle_FinishOperation(e);
}
