/*
 * FragmentController.hpp
 *
 *  Created on: Nov 27, 2011
 *      Author: heber
 */

#ifndef FRAGMENTCONTROLLER_HPP_
#define FRAGMENTCONTROLLER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/asio.hpp>
#include <string>
#include <vector>

#include "Connection.hpp"
#include "FragmentJob.hpp"
#include "FragmentResult.hpp"

#include "Controller/Commands/Operation.hpp"
#include "Controller/Commands/CheckResultsOperation.hpp"
#include "Controller/Commands/ReceiveJobsOperation.hpp"
#include "Controller/Commands/SendResultsOperation.hpp"
#include "Controller/Commands/ShutdownOperation.hpp"

/** The FragmentController sends bunches of jobs to a FragmentScheduler,
 *  waits for their calculation and is called when they are done. Then,
 *  he loads the bunch of results from the Scheduler.
 *
 * While the FragmentScheduler and FragmentWorker rather act on their own
 * this is the piece to implant into the user software to allow for
 * communication with the Server/Worker duo to perform the calculation
 * of the fragments on distant computers.
 */
class FragmentController
{
public:
  FragmentController(boost::asio::io_service& io_service, const std::string& _host, const std::string& _service);
  ~FragmentController();

protected:
  /// The Connection to the server.
  Connection connection_;

public:

  ReceiveJobsOperation recjobs;
  CheckResultsOperation checkres;
  SendResultsOperation  sendres;
  ShutdownOperation shutdown;

  // get the exit flag of the last operations
  size_t getExitflag() const
  {
    if (recjobs.getExitflag() != 0)
      return recjobs.getExitflag();
    if (checkres.getExitflag() != 0)
      return checkres.getExitflag();
    if (sendres.getExitflag() != 0)
      return sendres.getExitflag();
    if (shutdown.getExitflag() != 0)
      return shutdown.getExitflag();
    return 0;
  }

  /// place number of jobs into this controller
  void addJobs(const std::vector<FragmentJob> &jobs)
  {
    recjobs.addJobs(jobs);
  }

  /// get the results for the current jobs
  std::vector<FragmentResult> getResults()
  {
    return sendres.getResults();
  }

  /// Getter for doneJobs
  size_t getDoneJobs() const
  {
    return checkres.getDoneJobs();
  }

  /// Getter for number of jobs in the queue
  size_t getPresentJobs() const
  {
    return recjobs.getPresentJobs();
  }

private:
  /// host name of server
  const std::string host;

  // service to connect to to
  const std::string service;
};

#endif /* FRAGMENTCONTROLLER_HPP_ */
