/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CheckResultsOperation.cpp
 *
 *  Created on: Dec 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "ControllerChoices.hpp"

#include "Controller/Commands/CheckResultsOperation.hpp"

/** Handle connect operation to send number of done jobs.
 *
 * \param e error code if something went wrong
 * \param endpoint_iterator endpoint of the connection
 */
void CheckResultsOperation::handle_connect_check(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Give choice.
    enum ControllerChoices choice = CheckState;
    connection_.async_write(choice,
      boost::bind(&CheckResultsOperation::handle_ReceiveDoneJobs, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&CheckResultsOperation::handle_connect_check, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }
}

/** Callback function when doneJobs have been received.
 *
 * \param e error code if something went wrong
 */
void CheckResultsOperation::handle_ReceiveDoneJobs(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // The connection::async_read() function will automatically
    // decode the data that is written to the underlying socket.
    LOG(1, "INFO: Checking number of done jobs ...");
    connection_.async_read(doneJobs,
      boost::bind(&Operation::handle_FinishOperation, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }
}

/** Internal function to connect to the endpoint of the server asynchronuously.
 *
 * We require internal connetion_ and host and service to be set up for this.
 */
void CheckResultsOperation::connect_check()
{
  Info info(__FUNCTION__);
  // Resolve the host name into an IP address.
  boost::asio::ip::tcp::resolver::iterator endpoint_iterator = getEndpointIterator();
  boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;

  // Start an asynchronous connect operation.
  std::cout << "Connecting to endpoint " << endpoint << " to check " << std::endl;
  connection_.socket().async_connect(endpoint,
    boost::bind(&CheckResultsOperation::handle_connect_check, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
}

/** Prepares the calculation of the results for the current jobs.
 */
void CheckResultsOperation::operator()()
{
  Info info(__FUNCTION__);
  // connect
  connect_check();
  //disconnect
  disconnect();
}
