/** \file parsing.cpp
 *
 * Declarations of various class functions for the parsing of value files.
 *    
 */

using namespace std;

// ======================================= INCLUDES ==========================================

#include "helpers.hpp" 
#include "parser.hpp"

// ======================================= FUNCTIONS ==========================================

/** Test if given filename can be opened.
 * \param filename name of file
 * \return given file exists
 */
#ifdef HAVE_INLINE
inline
#endif
bool FilePresent(const char *filename)
{
  ifstream input;
  
  input.open(filename, ios::in);
  if (input == NULL) {
    cout << endl << "Unable to open " << filename << ", is the directory correct?" << endl;
    return false;
  }
  input.close();
  return true;
};

/** Test the given parameters.
 * \param argc argument count
 * \param **argv arguments array
 * \return given inputdir is valid
 */
#ifdef HAVE_INLINE
inline
#endif
bool TestParams(int argc, char **argv)
{
  ifstream input;
  stringstream line;

  line << argv[1] << FactorsFile;
  return FilePresent(line.str().c_str());
};

// ======================================= CLASS MatrixContainer =============================

/** Constructor of MatrixContainer class.
 */
MatrixContainer::MatrixContainer() {
  Matrix = NULL;
  Indices = NULL;
  Header = NULL;
  MatrixCounter = 0;
  RowCounter = NULL;
  ColumnCounter = 0;
};

/** Destructor of MatrixContainer class.
 */
MatrixContainer::~MatrixContainer() {
  if (Matrix != NULL)
    for(int i=0;i<MatrixCounter;i++) {
      if (RowCounter != NULL)
        for(int j=0;j<=RowCounter[i];j++)
          Free((void **)&Matrix[i][j], "MatrixContainer::~MatrixContainer: *Matrix[][]");
      Free((void **)&Matrix[i], "MatrixContainer::~MatrixContainer: **Matrix[]");
    }
  if ((RowCounter != NULL) && (Matrix[MatrixCounter] != NULL))
    for(int j=0;j<=RowCounter[MatrixCounter];j++)
      Free((void **)&Matrix[MatrixCounter][j], "MatrixContainer::~MatrixContainer: *Matrix[MatrixCounter][]");
  Free((void **)&Matrix[MatrixCounter], "MatrixContainer::~MatrixContainer: **Matrix[MatrixCounter]");
  Free((void **)&Matrix, "MatrixContainer::~MatrixContainer: ***Matrix");
  
  if (Indices != NULL)
    for(int i=0;i<=MatrixCounter;i++) {
      Free((void **)&Indices[i], "MatrixContainer::~MatrixContainer: *Indices[]");
    }
  Free((void **)&Indices, "MatrixContainer::~MatrixContainer: **Indices"); 
  
  Free((void **)&Header, "MatrixContainer::~MatrixContainer: *Header");
  Free((void **)&RowCounter, "MatrixContainer::~MatrixContainer: *RowCounter");
};

/** Parsing a number of matrices.
 * \param *name directory with files
 * \param *prefix prefix of each matrix file
 * \param *suffix suffix of each matrix file
 * \param skiplines number of inital lines to skip
 * \param skiplines number of inital columns to skip
 * \return parsing successful 
 */
bool MatrixContainer::ParseMatrix(char *name, char *prefix, char *suffix, int skiplines, int skipcolumns)
{
  char filename[1023];
  ifstream input;
  char *FragmentNumber = NULL;
  stringstream line;
  double tmp;  
  
  Header = (char *) Malloc(sizeof(char)*1023, "MatrixContainer::ParseMatrix: *EnergyHeader");

  // count the number of matrices
  MatrixCounter = -1; // we count one too much
  line << name << FactorsFile;
  input.open(line.str().c_str(), ios::in);
  if (input == NULL) {
    cout << endl << "Unable to open " << line.str() << ", is the directory correct?" << endl;
    return false;
  }
  while (!input.eof()) {
    input >> tmp;
    MatrixCounter++;
  }
  input.close();
  cout << "Determined " << MatrixCounter << " fragments." << endl;
  
  cout << "Parsing through each fragment and retrieving " << prefix << suffix << "." << endl;
  Matrix = (double ***) Malloc(sizeof(double **)*(MatrixCounter+1), "MatrixContainer::ParseMatrix: ***Matrix"); // one more each for the total molecule
  RowCounter = (int *) Malloc(sizeof(int)*(MatrixCounter+1), "MatrixContainer::ParseMatrix: *RowCounter");
  for(int i=0;i<=MatrixCounter;i++) {
    // energy
    stringstream line;
    FragmentNumber = FixedDigitNumber(MatrixCounter, i);
    if (i != MatrixCounter)  
      line << name << FragmentPrefix << FragmentNumber << "/" << prefix << suffix;
    else
      line << name << prefix << suffix;
    Free((void **)&FragmentNumber, "MatrixContainer::ParseMatrix: *FragmentNumber");
    input.open(line.str().c_str(), ios::in);
    //cout << "Opening " << line.str() << " ... "  << input << endl;
    if (input == NULL) {
      cerr << endl << "Unable to open " << line.str() << ", is the directory correct?" << endl;
      return false;
    }
    // scan header for number of columns
    for (int m=0;m<=skiplines;m++)
      input.getline(Header, 1023);
    line.str(Header);
    for(int k=0;k<skipcolumns;k++)
      line >> Header;
    //cout << line.str() << endl;
    ColumnCounter=0;
    while (!line.eof()) {   
      strcpy(filename,"@");
      line >> filename;
      if (filename[0] != '@')
        ColumnCounter++;
    }
    //cout << line.str() << endl;
    //cout << "ColumnCounter: " << ColumnCounter << endl;
    // scan rest for number of rows/lines
    RowCounter[i]=-1;    // counts one line too much
    while (!input.eof()) { 
      input.getline(filename, 1023);
      //cout << "Comparing: " << strncmp(filename,"MeanForce",9) << endl;
      RowCounter[i]++; // then line was not last MeanForce
      if (strncmp(filename,"MeanForce", 9) == 0) {
        break;
      }
    }
    cout << "RowCounter[" << i << "]: " << RowCounter[i] << endl;
    Matrix[i] = (double **) Malloc(sizeof(double *)*(RowCounter[i]+1), "MatrixContainer::ParseMatrix: **Matrix[]");
    input.clear();
    input.seekg(ios::beg);
    for (int m=0;m<=skiplines;m++)
      input.getline(Header, 1023);    // skip header
    line.str(Header);
    for(int k=0;k<skipcolumns;k++)  // skip columns in header too
      line >> filename;
    strncpy(Header, line.str().c_str(), 1023);  
    for(int j=0;j<RowCounter[i];j++) {
      Matrix[i][j] = (double *) Malloc(sizeof(double)*ColumnCounter, "MatrixContainer::ParseMatrix: *Matrix[][]");
      input.getline(filename, 1023);
      stringstream line(filename);
      //cout << "Matrix at level " << j << ":";// << filename << endl;
      for(int k=0;k<skipcolumns;k++)
        line >> filename;
      for(int k=0;(k<ColumnCounter) && (!line.eof());k++) {
        line >> Matrix[i][j][k];
        //cout << " " << setprecision(2) << Matrix[i][j][k];;
      }
      //cout << endl;
    }
    Matrix[i][ RowCounter[i] ] = (double *) Malloc(sizeof(double)*ColumnCounter, "MatrixContainer::ParseMatrix: *Matrix[RowCounter[i]][]");
    for(int j=0;j<ColumnCounter;j++)
      Matrix[i][ RowCounter[i] ][j] = 0.;
    input.close();
  }
  return true;
};

/** Allocates and resets the memory for a number \a MCounter of matrices.
 * \param *GivenHeader Header line
 * \param MCounter number of matrices
 * \param *RCounter number of rows for each matrix
 * \param CCounter number of columns for all matrices
 * \return Allocation successful 
 */
bool MatrixContainer::AllocateMatrix(char *GivenHeader, int MCounter, int *RCounter, int CCounter)
{
  Header = (char *) Malloc(sizeof(char)*1024, "MatrixContainer::ParseMatrix: *EnergyHeader");
  strncpy(Header, GivenHeader, 1023);

  MatrixCounter = MCounter;
  ColumnCounter = CCounter;
  Matrix = (double ***) Malloc(sizeof(double **)*(MatrixCounter+1), "MatrixContainer::ParseMatrix: ***Matrix"); // one more each for the total molecule
  RowCounter = (int *) Malloc(sizeof(int)*(MatrixCounter+1), "MatrixContainer::ParseMatrix: *RowCounter");
  for(int i=0;i<=MatrixCounter;i++) {
    RowCounter[i] = RCounter[i];
    Matrix[i] = (double **) Malloc(sizeof(double *)*(RowCounter[i]+1), "MatrixContainer::ParseMatrix: **Matrix[]");  
    for(int j=0;j<=RowCounter[i];j++) {
      Matrix[i][j] = (double *) Malloc(sizeof(double)*ColumnCounter, "MatrixContainer::ParseMatrix: *Matrix[][]");
      for(int k=0;k<ColumnCounter;k++)
        Matrix[i][j][k] = 0.;
    }
  }
  return true;
};

/** Resets all values in MatrixContainer::Matrix.
 * \return true if successful
 */
bool MatrixContainer::ResetMatrix()
{
  for(int i=0;i<=MatrixCounter;i++)
    for(int j=0;j<=RowCounter[i];j++)
      for(int k=0;k<ColumnCounter;k++)
        Matrix[i][j][k] = 0.;
   return true;
};

/** Sets all values in the last of MatrixContainer::Matrix to \a value.
 * \param value reset value
 * \param skipcolumns skip initial columns
 * \return true if successful
 */
bool MatrixContainer::SetLastMatrix(double value, int skipcolumns)
{
  for(int j=0;j<=RowCounter[MatrixCounter];j++)
    for(int k=skipcolumns;k<ColumnCounter;k++)
      Matrix[MatrixCounter][j][k] = value;
   return true;
};

/** Sets all values in the last of MatrixContainer::Matrix to \a value.
 * \param **values matrix with each value (must have at least same dimensions!)
 * \param skipcolumns skip initial columns
 * \return true if successful
 */
bool MatrixContainer::SetLastMatrix(double **values, int skipcolumns)
{
  for(int j=0;j<=RowCounter[MatrixCounter];j++)
    for(int k=skipcolumns;k<ColumnCounter;k++)
      Matrix[MatrixCounter][j][k] = values[j][k];
   return true;
};

/** Sums the energy with each factor and put into last element of \a ***Energies.
 * \param Matrix MatrixContainer with matrices (LevelCounter by ColumnCounter) with all the energies.
 * \param KeySet KeySetContainer with bond Order and association mapping of each fragment to an order
 * \param Order bond order
 * \return true if summing was successful
 */
bool MatrixContainer::SumSubManyBodyTerms(class MatrixContainer &MatrixValues, class KeySetsContainer &KeySet, int Order)
{
  // go through each order
  for (int CurrentFragment=0;CurrentFragment<KeySet.FragmentsPerOrder[Order];CurrentFragment++) {
    //cout << "Current Fragment is " << CurrentFragment << "/" << KeySet.OrderSet[Order][CurrentFragment] << "." << endl;
    // then go per order through each suborder and pick together all the terms that contain this fragment
    for(int SubOrder=0;SubOrder<=Order;SubOrder++) { // go through all suborders up to the desired order
      for (int j=0;j<KeySet.FragmentsPerOrder[SubOrder];j++) { // go through all possible fragments of size suborder
        if (KeySet.Contains(KeySet.OrderSet[Order][CurrentFragment], KeySet.OrderSet[SubOrder][j])) {
          //cout << "Current other fragment is " << j << "/" << KeySet.OrderSet[SubOrder][j] << "." << endl;
          // if the fragment's indices are all in the current fragment
          for(int k=0;k<RowCounter[ KeySet.OrderSet[SubOrder][j] ];k++) { // go through all atoms in this fragment
            int m = MatrixValues.Indices[ KeySet.OrderSet[SubOrder][j] ][k];
            //cout << "Current index is " << k << "/" << m << "." << endl;
            if (m != -1) { // if it's not an added hydrogen
              for (int l=0;l<RowCounter[ KeySet.OrderSet[Order][CurrentFragment] ];l++) { // look for the corresponding index in the current fragment
                //cout << "Comparing " << m << " with " << Matrix.Indices[ KeySet.OrderSet[Order][CurrentFragment] ][l] << "." << endl;
                if (m == MatrixValues.Indices[ KeySet.OrderSet[Order][CurrentFragment] ][l]) {
                  m = l;
                  break;  
                }
              }
              //cout << "Corresponding index in CurrentFragment is " << m << "." << endl;
              if (m > RowCounter[ KeySet.OrderSet[Order][CurrentFragment] ]) {
                cerr << "Current force index " << m << " is greater than " << RowCounter[ KeySet.OrderSet[Order][CurrentFragment] ] << "!" << endl;
                return false;
              }
              if (Order == SubOrder) { // equal order is always copy from Energies
                for(int l=0;l<ColumnCounter;l++) // then adds/subtract each column 
                  Matrix[ KeySet.OrderSet[Order][CurrentFragment] ][m][l] += MatrixValues.Matrix[ KeySet.OrderSet[SubOrder][j] ][k][l];
              } else {
                for(int l=0;l<ColumnCounter;l++)
                  Matrix[ KeySet.OrderSet[Order][CurrentFragment] ][m][l] -= Matrix[ KeySet.OrderSet[SubOrder][j] ][k][l];
              }
            }
          //if ((ColumnCounter>1) && (RowCounter >= 1))
            //cout << "Fragments[ KeySet.OrderSet[" << i << "][" << CurrentFragment << "]=" << KeySet.OrderSet[Order][CurrentFragment] << " ][" << RowCounter[0]-1 << "][" << 1 << "] = " <<  Fragments[ KeySet.OrderSet[Order][CurrentFragment] ][RowCounter[0]-1][1] << endl;
          }
        } else {
          //cout << "Fragment " << KeySet.OrderSet[SubOrder][j] << " is not contained in fragment " << KeySet. KeySet.OrderSet[Order][CurrentFragment] << "." << endl;
        }
      }
    }
    //cout << "Final Fragments[ KeySet.OrderSet[" << Order << "][" << CurrentFragment << "]=" << KeySet.OrderSet[Order][CurrentFragment] << " ][" << KeySet.AtomCounter[0]-1 << "][" << 1 << "] = " <<  Fragments[ KeySet.OrderSet[Order][CurrentFragment] ][KeySet.AtomCounter[0]-1][1] << endl;
  }
  
  return true;
};

/** Writes the summed total fragment terms \f$F_{ij}\f$ to file.
 * \param *name inputdir
 * \param *prefix prefix before \a EnergySuffix
 * \return file was written
 */
bool MatrixContainer::WriteTotalFragments(const char *name, const char *prefix)
{
  ofstream output;
  char *FragmentNumber = NULL;

  cout << "Writing fragment files." << endl;
  for(int i=0;i<MatrixCounter;i++) {
    stringstream line;
    FragmentNumber = FixedDigitNumber(MatrixCounter, i);
    line << name << FragmentPrefix << FragmentNumber << "/" << prefix;
    Free((void **)&FragmentNumber, "*FragmentNumber");
    output.open(line.str().c_str(), ios::out);
    if (output == NULL) {
      cerr << "Unable to open output energy file " << line.str() << "!" << endl;
      return false;
    }
    output << Header << endl;
    for(int j=0;j<RowCounter[i];j++) {
      for(int k=0;k<ColumnCounter;k++)
        output << scientific << Matrix[i][j][k] << "\t";
      output << endl;
    }
    output.close();
  }
  return true;
};

/** Writes the summed total values in the last matrix to file.
 * \param *name inputdir
 * \param *prefix prefix
 * \param *suffix suffix
 * \return file was written
 */
bool MatrixContainer::WriteLastMatrix(const char *name, const char *prefix, const char *suffix)
{
  ofstream output;
  stringstream line;

  cout << "Writing matrix values of " << suffix << "." << endl;
  line << name << prefix << suffix;
  output.open(line.str().c_str(), ios::out);
  if (output == NULL) {
    cerr << "Unable to open output matrix file " << line.str() << "!" << endl;
    return false;
  }
  output << Header << endl;
  for(int j=0;j<RowCounter[MatrixCounter];j++) {
    for(int k=0;k<ColumnCounter;k++)
      output << scientific << Matrix[MatrixCounter][j][k] << "\t";
    output << endl;
  }
  output.close();
  return true;
};

// ======================================= CLASS EnergyMatrix =============================

/** Create a trivial energy index mapping.
 * This just maps 1 to 1, 2 to 2 and so on for all fragments.
 * \return creation sucessful
 */
bool EnergyMatrix::ParseIndices() 
{
  cout << "Parsing energy indices." << endl;
  Indices = (int **) Malloc(sizeof(int *)*(MatrixCounter+1), "EnergyMatrix::ParseIndices: **Indices");
  for(int i=0;i<=MatrixCounter;i++) {
    Indices[i] = (int *) Malloc(sizeof(int)*RowCounter[i], "EnergyMatrix::ParseIndices: *Indices[]");
    for(int j=0;j<RowCounter[i];j++)
      Indices[i][j] = j;
  } 
  return true; 
};

/** Sums the energy with each factor and put into last element of \a EnergyMatrix::Matrix.
 * \param Matrix MatrixContainer with matrices (LevelCounter by ColumnCounter) with all the energies.
 * \param KeySet KeySetContainer with bond Order and association mapping of each fragment to an order
 * \param Order bond order
 * \return true if summing was successful
 */
bool EnergyMatrix::SumSubEnergy(class EnergyMatrix &Fragments, class KeySetsContainer &KeySet, int Order)
{
  // sum energy
  for(int i=0;i<KeySet.FragmentsPerOrder[Order];i++)
    for(int j=0;j<RowCounter[ KeySet.OrderSet[Order][i] ];j++)
      for(int k=0;k<ColumnCounter;k++)
        Matrix[MatrixCounter][j][k] += Fragments.Matrix[ KeySet.OrderSet[Order][i] ][j][k];
  return true;
};

// ======================================= CLASS ForceMatrix =============================

/** Parsing force Indices of each fragment
 * \param *name directory with \a ForcesFile
 * \return parsing successful 
 */
bool ForceMatrix::ParseIndices(char *name) 
{
  ifstream input;
  char *FragmentNumber = NULL;
  char filename[1023];
  stringstream line;
  
  cout << "Parsing force indices." << endl;
  Indices = (int **) Malloc(sizeof(int *)*(MatrixCounter+1), "ForceMatrix::ParseIndices: **Indices");
  line << name << ForcesFile;
  input.open(line.str().c_str(), ios::in);
  //cout << "Opening " << line.str() << " ... "  << input << endl;
  if (input == NULL) {
    cout << endl << "Unable to open " << line.str() << ", is the directory correct?" << endl;
    return false;
  }
  for (int i=0;(i<MatrixCounter) && (!input.eof());i++) {
    // get the number of atoms for this fragment
    stringstream line;
    input.getline(filename, 1023);
    line.str(filename);
    // parse the values
    Indices[i] = (int *) Malloc(sizeof(int)*RowCounter[i], "ForceMatrix::ParseIndices: *Indices[]");
    FragmentNumber = FixedDigitNumber(MatrixCounter, i);
    cout << FragmentPrefix << FragmentNumber << "[" << RowCounter[i] << "]:";
    Free((void **)&FragmentNumber, "ForceMatrix::ParseIndices: *FragmentNumber");
    for(int j=0;(j<RowCounter[i]) && (!line.eof());j++) {
      line >> Indices[i][j];
      cout << " " << Indices[i][j];
    }
    cout << endl;
  }
  Indices[MatrixCounter] = (int *) Malloc(sizeof(int)*RowCounter[MatrixCounter], "ForceMatrix::ParseIndices: *Indices[]");
  for(int j=0;j<RowCounter[MatrixCounter];j++) {
    Indices[MatrixCounter][j] = j;
  }
  input.close();
  return true;
};


/** Sums the forces and puts into last element of \a ForceMatrix::Matrix.
 * \param Matrix MatrixContainer with matrices (LevelCounter by ColumnCounter) with all the energies.
 * \param KeySet KeySetContainer with bond Order and association mapping of each fragment to an order
 * \param Order bond order
 * \return true if summing was successful
 */
bool ForceMatrix::SumSubForces(class ForceMatrix &Fragments, class KeySetsContainer &KeySet, int Order)
{
  // sum forces
  for(int i=0;i<KeySet.FragmentsPerOrder[Order];i++)
    for(int l=0;l<RowCounter[ KeySet.OrderSet[Order][i] ];l++) {
      int j = Indices[ KeySet.OrderSet[Order][i] ][l];
      if (j > RowCounter[MatrixCounter]) {
        cerr << "Current force index " << j << " is greater than " << RowCounter[MatrixCounter] << "!" << endl;
        return false;
      }
      if (j != -1)
        for(int k=2;k<ColumnCounter;k++)
          Matrix[MatrixCounter][j][k] += Fragments.Matrix[ KeySet.OrderSet[Order][i] ][l][k];
    }
  return true;
};


// ======================================= CLASS KeySetsContainer =============================

/** Constructor of KeySetsContainer class.
 */
KeySetsContainer::KeySetsContainer() {
  KeySets = NULL;
  AtomCounter = NULL;
  FragmentCounter = 0;
  Order = 0;
  FragmentsPerOrder = 0;
  OrderSet = NULL;
};

/** Destructor of KeySetsContainer class.
 */
KeySetsContainer::~KeySetsContainer() {
  for(int i=0;i<FragmentCounter;i++)
    Free((void **)&KeySets[i], "KeySetsContainer::~KeySetsContainer: *KeySets[]");
  for(int i=0;i<Order;i++)
    Free((void **)&OrderSet[i], "KeySetsContainer::~KeySetsContainer: *OrderSet[]");
  Free((void **)&KeySets, "KeySetsContainer::~KeySetsContainer: **KeySets");
  Free((void **)&OrderSet, "KeySetsContainer::~KeySetsContainer: **OrderSet");
  Free((void **)&AtomCounter, "KeySetsContainer::~KeySetsContainer: *AtomCounter");
  Free((void **)&FragmentsPerOrder, "KeySetsContainer::~KeySetsContainer: *FragmentsPerOrder");
};

/** Parsing KeySets into array.
 * \param *name directory with keyset file
 * \param *ACounter number of atoms per fragment
 * \param FCounter number of fragments
 * \return parsing succesful
 */
bool KeySetsContainer::ParseKeySets(const char *name, const int *ACounter, const int FCounter) {
  ifstream input;
  char *FragmentNumber = NULL;
  stringstream line;
  char filename[1023];
  
  FragmentCounter = FCounter;
  cout << "Parsing key sets." << endl;
  KeySets = (int **) Malloc(sizeof(int *)*FragmentCounter, "KeySetsContainer::ParseKeySets: **KeySets");
  line << name << KeySetsFile;
  input.open(line.str().c_str(), ios::in);
  if (input == NULL) {
    cout << endl << "Unable to open " << line.str() << ", is the directory correct?" << endl;
    return false;
  }
  
  AtomCounter = (int *) Malloc(sizeof(int)*FragmentCounter, "KeySetsContainer::ParseKeySets: *RowCounter");
  for(int i=0;(i<FragmentCounter) && (!input.eof());i++) {
    stringstream line;
    AtomCounter[i] = ACounter[i];
    // parse the values
    KeySets[i] = (int *) Malloc(sizeof(int)*AtomCounter[i], "KeySetsContainer::ParseKeySets: *KeySets[]");
    FragmentNumber = FixedDigitNumber(FragmentCounter, i);
    cout << FragmentPrefix << FragmentNumber << "[" << AtomCounter[i] << "]:";
    Free((void **)&FragmentNumber, "KeySetsContainer::ParseKeySets: *FragmentNumber");
    input.getline(filename, 1023);
    line.str(filename);
    for(int j=0;(j<AtomCounter[i]) && (!line.eof());j++) {
      line >> KeySets[i][j];
      cout << " " << KeySets[i][j];
    }
    cout << endl;
  }
  input.close();
  return true;
};

/** Parse many body terms, associating each fragment to a certain bond order.
 * \return parsing succesful
 */
bool KeySetsContainer::ParseManyBodyTerms()
{
  int Counter;
  
  cout << "Creating Fragment terms." << endl;
  // scan through all to determine order
  Order=0;
  for(int i=0;i<FragmentCounter;i++) {
    Counter=0;
    for(int j=0;j<AtomCounter[i];j++)
      if (KeySets[i][j] != -1)
        Counter++;
    if (Counter > Order)
      Order = Counter;
  }
  cout << "Found Order is " << Order << "." << endl;
  
  // scan through all to determine fragments per order
  FragmentsPerOrder = (int *) Malloc(sizeof(int)*Order, "KeySetsContainer::ParseManyBodyTerms: *FragmentsPerOrder");
  for(int i=0;i<Order;i++)
    FragmentsPerOrder[i] = 0;
  for(int i=0;i<FragmentCounter;i++) {
    Counter=0;
    for(int j=0;j<AtomCounter[i];j++)
      if (KeySets[i][j] != -1)
        Counter++;
    FragmentsPerOrder[Counter-1]++;
  }
  for(int i=0;i<Order;i++)
    cout << "Found No. of Fragments of Order " << i+1 << " is " << FragmentsPerOrder[i] << "." << endl;
    
  // scan through all to gather indices to each order set
  OrderSet = (int **) Malloc(sizeof(int *)*Order, "KeySetsContainer::ParseManyBodyTerms: **OrderSet");
  for(int i=0;i<Order;i++)
    OrderSet[i] = (int *) Malloc(sizeof(int)*FragmentsPerOrder[i], "KeySetsContainer::ParseManyBodyTermsKeySetsContainer::ParseManyBodyTerms: *OrderSet[]");
  for(int i=0;i<Order;i++)
    FragmentsPerOrder[i] = 0;
  for(int i=0;i<FragmentCounter;i++) {
    Counter=0;
    for(int j=0;j<AtomCounter[i];j++)
      if (KeySets[i][j] != -1)
        Counter++;
    OrderSet[Counter-1][FragmentsPerOrder[Counter-1]] = i;
    FragmentsPerOrder[Counter-1]++;
  }
  cout << "Printing OrderSet." << endl;
  for(int i=0;i<Order;i++) {
    for (int j=0;j<FragmentsPerOrder[i];j++) {
      cout << " " << OrderSet[i][j];
    }
    cout << endl;
  }
  cout << endl;
  
    
  return true;
};

/** Compares each entry in \a *SmallerSet if it is containted in \a *GreaterSet.
 * \param GreaterSet index to greater set
 * \param SmallerSet index to smaller set
 * \return true if all keys of SmallerSet contained in GreaterSet
 */
bool KeySetsContainer::Contains(const int GreaterSet, const int SmallerSet)
{
  bool result = true;
  bool intermediate;
  if ((GreaterSet < 0) || (SmallerSet < 0) || (GreaterSet > FragmentCounter) || (SmallerSet > FragmentCounter)) // index out of bounds
    return false;
  for(int i=0;i<AtomCounter[SmallerSet];i++) {
    intermediate = false;
    for (int j=0;j<AtomCounter[GreaterSet];j++)
      intermediate = (intermediate || ((KeySets[SmallerSet][i] == KeySets[GreaterSet][j]) || (KeySets[SmallerSet][i] == -1)));
    result = result && intermediate;
  }

  return result;
};


// ======================================= END =============================================
