/** \file joiner.cpp
 *
 * Takes evaluated fragments (energy and forces) and by reading the factors files determines total energy
 * and each force for the whole molecule. 
 *    
 */

//============================ INCLUDES ===========================

#include "helpers.hpp" 
#include "parser.hpp" 

//============================== MAIN =============================

int main(int argc, char **argv)
{
  EnergyMatrix Energy;
  ForceMatrix Force;
  EnergyMatrix EnergyFragments;
  ForceMatrix ForceFragments;
  KeySetsContainer KeySet;  
  stringstream prefix;

  cout << "Joiner" << endl;
  cout << "======" << endl;
  
  // Get the command line options
  if (argc < 3) {
    cout << "Usage: " << argv[0] << " <inputdir> <prefix>" << endl;
    cout << "<inputdir>\ttherein the output of a molecuilder fragmentation is expected, each fragment with a subdir containing an energy.all and a forces.all file." << endl;
    cout << "<prefix>\tprefix of energy and forces file." << endl;
    return 1;
  }
  
  // Test the given directory
  if (!TestParams(argc, argv))
    return 1;
  
  // +++++++++++++++++ PARSING +++++++++++++++++++++++++++++++
  
  // ------------- Parse through all Fragment subdirs --------
  if (!Energy.ParseMatrix(argv[1], argv[2], EnergySuffix,0,0)) return 1;
  //if (!ParseSubEnergies(argv[1], argv[2], EnergySuffix, data.EnergyHeader, data.Energies, data.EnergyIndices, data.LevelCounter, data.ColumnCounter, data.FragmentCounter)) return 1;
  if (!Force.ParseMatrix(argv[1], argv[2], ForcesSuffix, 0,0)) return 1;
  //if (!ParseSubForces(argv[1], argv[2], ForcesSuffix, data.ForcesHeader, data.Forces, data.AtomCounter, data.Column2Counter, data.FragmentCounter)) return 1;

  // ---------- Parse the TE Factors into an array -----------------
  if (!Energy.ParseIndices()) return 1;
  
  // ---------- Parse the Force indices into an array ---------------
  if (!Force.ParseIndices(argv[1])) return 1;
  //if (!ParseForceIndices(argv[1], data.ForcesIndices, data.AtomCounter, data.FragmentCounter)) return 1;

  // ---------- Parse the KeySets into an array ---------------
  if (!KeySet.ParseKeySets(argv[1], Force.RowCounter, Force.MatrixCounter)) return 1;
  //if (!ParseKeySets(argv[1], data.KeySets, data.AtomCounter, data.FragmentCounter)) return 1;

  if (!KeySet.ParseManyBodyTerms()) return 1;
  //if (!ParseManyBodyTerms(data.Order, data.OrderSet, data.FragmentsPerOrder, data.KeySets, data.AtomCounter, data.FragmentCounter)) return 1;
  if (!EnergyFragments.AllocateMatrix(Energy.Header, Energy.MatrixCounter, Energy.RowCounter, Energy.ColumnCounter)) return 1;
  //if (!ParseSubEnergies(argv[1], argv[2], EnergyFragmentSuffix, data.EnergyHeader, data.EnergyFragments, data.EnergyFragmentIndices, data.LevelFragmentCounter, data.ColumnFragmentCounter, data.FragmentCounter)) return 1;
  if (!ForceFragments.AllocateMatrix(Force.Header, Force.MatrixCounter, Force.RowCounter, Force.ColumnCounter)) return 1;
  //if (!ParseSubForces(argv[1], argv[2], ForceFragmentSuffix, data.ForcesHeader, data.ForceFragments, data.AtomCounter, data.Column2Counter, data.FragmentCounter)) return 1;
  
  // ----------- Resetting last matrices (where full QM values are stored right now)
  if(!Energy.SetLastMatrix(0., 0)) return 1;
  if(!Force.SetLastMatrix(0., 2)) return 1;
  
  // +++++++++++++++++ SUMMING +++++++++++++++++++++++++++++++

  // --------- sum up and write for each order----------------
  for (int BondOrder=0;BondOrder<KeySet.Order;BondOrder++) {
    // --------- sum up energy --------------------
    cout << "Summing energy of order " << BondOrder+1 << " ..." << endl;
    if (!EnergyFragments.SumSubManyBodyTerms(Energy, KeySet, BondOrder)) return 1;
    if (!Energy.SumSubEnergy(EnergyFragments, KeySet, BondOrder)) return 1;
    // --------- sum up Forces --------------------
    cout << "Summing forces of order " << BondOrder+1 << " ..." << endl;
    if (!ForceFragments.SumSubManyBodyTerms(Force, KeySet, BondOrder)) return 1;
    if (!Force.SumSubForces(ForceFragments, KeySet, BondOrder)) return 1;

    // --------- write the energy and forces file --------------------
    prefix.str(" ");
    prefix << argv[2] << OrderSuffix << (BondOrder+1);
    cout << "Writing files " << argv[1] << prefix.str() << ". ..." << endl;
    // energy
    if (!Energy.WriteLastMatrix(argv[1], (prefix.str()).c_str(), EnergySuffix)) return 1;
    // forces
    if (!Force.WriteLastMatrix(argv[1], (prefix.str()).c_str(), ForcesSuffix)) return 1;
  }
  // fragments
  prefix.str(" ");
  prefix << argv[2] << EnergyFragmentSuffix;
  if (!EnergyFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
  prefix.str(" ");
  prefix << argv[2] << ForceFragmentSuffix;
  if (!ForceFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;

  // write last matrices as fragments into central dir (not subdir as above), for analyzer to know index bounds
  if (!Energy.WriteLastMatrix(argv[1], argv[2], EnergyFragmentSuffix)) return 1;
  if (!Force.WriteLastMatrix(argv[1], argv[2], ForceFragmentSuffix)) return 1;

  // exit  
  cout << "done." << endl;
  return 0;
};

//============================ END ===========================
