/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IndexSet.cpp
 *
 *  Created on: Apr 24, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "IndexSet.hpp"

#include <boost/foreach.hpp>

/** Constructor for class IndexSet.
 *
 */
IndexSet::IndexSet()
{}

/** Destructor for class IndexSet.
 *
 */
IndexSet::~IndexSet()
{
  clear();
}

/** Operator for output to std::ostream operator of an IndexSet.
 * @param ost output stream
 * @param indexset index set to output
 * @return ost output stream
 */
std::ostream & operator<<(std::ostream &ost, const IndexSet &indexset)
{
  ost << "{ ";
  for (IndexSet::const_iterator iter = indexset.begin();
      iter != indexset.end();
      ++iter)
    ost << *iter << " ";
  ost << "}";
  return ost;
}

/** Checks whether each index of _IndexSet is contained in this subspace:Indices.
 *
 * @param _indexset index set to check
 * @return true - _index is contained, false - at least index is not contained
 */
bool IndexSet::contains(const IndexSet &_indexset) const
{
  BOOST_FOREACH(size_t index, _indexset) {
    if (!contains(index)) {  // index not present, then not contained
      return false;
    }
  }
  return true;
}

/** Checks whether index is contained in this IndexSet.
 *
 * @param _index index to check
 * @return true - _index is contained, false - at least index is not contained
 */
bool IndexSet::contains(const Index_t _index) const
{
  const_iterator iter = find(_index);
  if (iter == end()) {  // index not present, then not contained
    return false;
  } else {
    return true;
  }
}

bool IndexSet::operator<(const IndexSet &b) const
{
  if (size() < b.size()) {
    return true;
  } else if (size() > b.size()) {
    return false;
  } else {
    const_iterator this_iter = begin();
    const_iterator b_iter = b.begin();
    for (;(this_iter != end()) && (b_iter != b.end());
        ++this_iter, ++b_iter) {
      if (*this_iter < *b_iter) {
        return true;
      } else if (*this_iter > *b_iter) {
        return false;
      }
    }
    return false;
  }
}

bool IndexSet::operator>(const IndexSet &b) const
{
  if (size() > b.size()) {
    return true;
  } else if (size() < b.size()) {
    return false;
  } else {
    const_iterator this_iter = begin();
    const_iterator b_iter = b.begin();
    for (;(this_iter != end()) && (b_iter != b.end());
        ++this_iter, ++b_iter) {
      if (*this_iter > *b_iter) {
        return true;
      } else if (*this_iter < *b_iter) {
        return false;
      }
    }
    return false;
  }
}

bool IndexSet::operator==(const IndexSet &b) const
{
  if (size() != b.size()) {
    return false;
  } else {
    const_iterator this_iter = begin();
    const_iterator b_iter = b.begin();
    for (;(this_iter != end()) && (b_iter != b.end());
        ++this_iter, ++b_iter) {
      if (*this_iter != *b_iter) {
        return false;
      }
    }
    return true;
  }
}

