/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FragmentJob.cpp
 *
 *  Created on: Oct 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cstdlib>
#include <fstream>
#include <string>

#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/toString.hpp"
#include "FragmentJob.hpp"
#include "FragmentResult.hpp"

/** Constructor for class FragmentJob.
 *
 */
FragmentJob::FragmentJob() :
  JobId(-1),
  outputfile("")
{}

/** Constructor for class FragmentJob.
 *
 * \param _command command to execute
 * \param _outputfile configuration file for solver
 * \param _JobId unique id of this job
 */
FragmentJob::FragmentJob(const std::string &_command, const std::string &_outputfile, const JobId_t _JobId) :
  JobId(_JobId),
  command(_command),
  outputfile(_outputfile)
{}

/** Destructor for class FragmentJob.
 *
 */
FragmentJob::~FragmentJob()
{}

/** Work routine of this FragmentJob.
 *
 * This function encapsulates all the work that has to be done to generate
 * a FragmentResult. Hence, the FragmentWorker does not need to know anything
 * about the operation: it just receives it and executes this function.
 *
 * \return result of this job
 */
FragmentResult FragmentJob::Work()
{
  Info info((std::string(__FUNCTION__)+std::string(", id #")+toString(getId())).c_str());
  FragmentResult s(getId());

  // write outputfile to file
  std::string inputfilename =
      std::string("/tmp/dummy_")
      +toString(getId());
  std::string outputfilename = inputfilename+std::string(".out");
  inputfilename += std::string(".dat");
  std::ofstream output(outputfilename.c_str());
  output << outputfile << std::endl;
  output.close();

  // fork into subprocess and launch command
  s.exitflag = std::system((command+std::string(" ")+inputfilename
                        +std::string(" >")+outputfilename).c_str());
  std::ifstream resultfile;
  resultfile.open(outputfilename.c_str(), ifstream::in);
  char line[1024];
  while (resultfile.good()) {
    resultfile.getline(line, 1023);
    s.result += line;
  }
  resultfile.close();

  return s;
}

/** Comparator for class FragmentJob.
 * \param other instance to compare to
 * \return every member variable is the same, else - is not
 */
bool FragmentJob::operator==(const FragmentJob &other) const
{
  return (command == other.command)
      && (outputfile == other.outputfile)
      && (getId() == other.getId());
}
