/*
 * gslmatrixunittest.cpp
 *
 *  Created on: Jan 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "gslmatrixunittest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( GSLMatrixTest );


void GSLMatrixTest::setUp()
{
  m = new GSLMatrix(4,3);
};

void GSLMatrixTest::tearDown()
{
  delete(m);
};

/** Unit Test for accessing matrix elements.
 *
 */
void GSLMatrixTest::AccessTest()
{
  // check whether all elements are initially zero
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( 0., m->Get(i,j) );

  // set
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      m->Set(i,j, i*3+j );

  // and check
  double *ptr = NULL;
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++) {
      CPPUNIT_ASSERT_EQUAL( (double)(i*3+j), m->Get(i,j) );
      ptr = m->Pointer(i,j);
      CPPUNIT_ASSERT_EQUAL( (double)(i*3+j), *ptr );
    }

  // assignment
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      m->Set(i,j, i*3+j );
  GSLMatrix *dest = new GSLMatrix(4,3);
  *dest = *m;
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( dest->Get(i,j), m->Get(i,j) );
  delete(dest);

  // out of bounds
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(5,2) );
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(2,17) );
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(1024,140040) );
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(-1,0) );
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(0,-1) );
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(-1,-1) );
};

/** Unit Test for initializating matrices.
 *
 */
void GSLMatrixTest::InitializationTest()
{
  // set zero
  m->SetZero();
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( 0., m->Get(i,j) );

  // set all
  m->SetAll(1.5);
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( 1.5, m->Get(i,j) );

  // set basis
  m->SetIdentity();
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( i == j ? 1. : 0. , m->Get(i,j) );

  // set from array
  double array[] = { 1., 0., 0.,
                     0., 1., 0.,
                     0., 0., 1.,
                     0., 0., 0. };
  m->SetFromDoubleArray(array);
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( i == j ? 1. : 0. , m->Get(i,j) );

};

/** Unit Test for copying matrices.
 *
 */
void GSLMatrixTest::CopyTest()
{
  // set basis
  GSLMatrix *dest = NULL;
  for (int i=0;i<4;i++) {
    m->SetAll(i);
    dest = new GSLMatrix(m);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( m->Get(i,j) , dest->Get(i,j) );

    delete(dest);
  }
};

/** Unit Test for exchanging rows and columns.
 *
 */
void GSLMatrixTest::ExchangeTest()
{
  // set to 1,1,1,2, ...
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      m->Set(i,j, i+1 );

  // swap such that nothing happens
  CPPUNIT_ASSERT_EQUAL( true, m->SwapColumns(1,2) );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (double)i+1., m->Get(i,j) );

  // swap two rows
  CPPUNIT_ASSERT_EQUAL( true, m->SwapRows(1,2) );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      switch (i) {
        case 0:
          CPPUNIT_ASSERT_EQUAL( 1., m->Get(i,j) );
          break;
        case 1:
          CPPUNIT_ASSERT_EQUAL( 3., m->Get(i,j) );
          break;
        case 2:
          CPPUNIT_ASSERT_EQUAL( 2., m->Get(i,j) );
          break;
        case 3:
          CPPUNIT_ASSERT_EQUAL( 4., m->Get(i,j) );
          break;
        default:
          CPPUNIT_ASSERT_EQUAL( -1., m->Get(i,j) );
      }
  // check that op is reversable
  CPPUNIT_ASSERT_EQUAL( true, m->SwapRows(1,2) );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (double)i+1., m->Get(i,j) );

  // set to 1,2,3,1, ...
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      m->Set(i,j, j+1. );

  // swap such that nothing happens
  CPPUNIT_ASSERT_EQUAL( true, m->SwapRows(0,2) );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (double)j+1., m->Get(i,j) );

  // swap two columns
  CPPUNIT_ASSERT_EQUAL( true, m->SwapColumns(0,2) );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      switch (j) {
        case 0:
          CPPUNIT_ASSERT_EQUAL( 3., m->Get(i,j) );
          break;
        case 1:
          CPPUNIT_ASSERT_EQUAL( 2., m->Get(i,j) );
          break;
        case 2:
          CPPUNIT_ASSERT_EQUAL( 1., m->Get(i,j) );
          break;
        default:
          CPPUNIT_ASSERT_EQUAL( -1., m->Get(i,j) );
      }
  // check that op is reversable
  CPPUNIT_ASSERT_EQUAL( true, m->SwapColumns(0,2) );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (double)j+1., m->Get(i,j) );


  // set to 1,2,3,4, ...
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      m->Set(i,j, 3*i+j+1 );
  // transpose
  CPPUNIT_ASSERT_EQUAL( true, m->Transpose() );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (double)(3*i+j+1), m->Get(j,i) );
  // second transpose
  CPPUNIT_ASSERT_EQUAL( true, m->Transpose() );
  for (int i=0;i<4;i++)
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (double)(3*i+j+1), m->Get(i,j) );
};

/** Unit Test for matrix properties.
 *
 */
void GSLMatrixTest::PropertiesTest()
{
  // is zero
  m->SetZero();
  CPPUNIT_ASSERT_EQUAL( true, m->IsNull() );
  CPPUNIT_ASSERT_EQUAL( false, m->IsPositive() );
  CPPUNIT_ASSERT_EQUAL( false, m->IsNegative() );
  CPPUNIT_ASSERT_EQUAL( true, m->IsNonNegative() );

  // is positive
  m->SetAll(0.5);
  CPPUNIT_ASSERT_EQUAL( false, m->IsNull() );
  CPPUNIT_ASSERT_EQUAL( true, m->IsPositive() );
  CPPUNIT_ASSERT_EQUAL( false, m->IsNegative() );
  CPPUNIT_ASSERT_EQUAL( true, m->IsNonNegative() );

  // is negative
  m->SetAll(-0.1);
  CPPUNIT_ASSERT_EQUAL( false, m->IsNull() );
  CPPUNIT_ASSERT_EQUAL( false, m->IsPositive() );
  CPPUNIT_ASSERT_EQUAL( true, m->IsNegative() );
  CPPUNIT_ASSERT_EQUAL( false, m->IsNonNegative() );

  // is positive definite
  CPPUNIT_ASSERT_EQUAL( false, m->IsPositiveDefinite() );
};
