/*
 * ParserUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Parser/MpqcParser.hpp"
#include "Parser/PcpParser.hpp"
#include "Parser/TremoloParser.hpp"
#include "Parser/XyzParser.hpp"
#include "World.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserUnitTest );

static string waterPcp = "# ParallelCarParinello - main configuration file - created with molecuilder\n\
\n\
mainname\tpcp\t# programm name (for runtime files)\n\
defaultpath\not specified\t# where to put files during runtime\n\
pseudopotpath\not specified\t# where to find pseudopotentials\n\
\n\
ProcPEGamma\t8\t# for parallel computing: share constants\n\
ProcPEPsi\t1\t# for parallel computing: share wave functions\n\
DoOutVis\t0\t# Output data for OpenDX\n\
DoOutMes\t1\t# Output data for measurements\n\
DoOutOrbitals\t0\t# Output all Orbitals\n\
DoOutCurr\t0\t# Ouput current density for OpenDx\n\
DoOutNICS\t0\t# Output Nucleus independent current shieldings\n\
DoPerturbation\t0\t# Do perturbation calculate and determine susceptibility and shielding\n\
DoFullCurrent\t0\t# Do full perturbation\n\
DoConstrainedMD\t0\t# Do perform a constrained (>0, relating to current MD step) instead of unconstrained (0) MD\n\
Thermostat\tBerendsen\t2.5\t# Which Thermostat and its parameters to use in MD case.\n\
CommonWannier\t0\t# Put virtual centers at indivual orbits, all common, merged by variance, to grid point, to cell center\n\
SawtoothStart\t0.01\t# Absolute value for smooth transition at cell border \n\
VectorPlane\t0\t# Cut plane axis (x, y or z: 0,1,2) for two-dim current vector plot\n\
VectorCut\t0\t# Cut plane axis value\n\
AddGramSch\t1\t# Additional GramSchmidtOrtogonalization to be safe\n\
Seed\t1\t# initial value for random seed for Psi coefficients\n\
\n\
MaxOuterStep\t0\t# number of MolecularDynamics/Structure optimization steps\n\
Deltat\t0.01\t# time per MD step\n\
OutVisStep\t10\t# Output visual data every ...th step\n\
OutSrcStep\t5\t# Output \"restart\" data every ..th step\n\
TargetTemp\t0.000950045\t# Target temperature\n\
MaxPsiStep\t3\t# number of Minimisation steps per state (0 - default)\n\
EpsWannier\t1e-07\t# tolerance value for spread minimisation of orbitals\n\
# Values specifying when to stop\n\
MaxMinStep\t100\t# Maximum number of steps\n\
RelEpsTotalE\t1e-07\t# relative change in total energy\n\
RelEpsKineticE\t1e-05\t# relative change in kinetic energy\n\
MaxMinStopStep\t2\t# check every ..th steps\n\
MaxMinGapStopStep\t1\t# check every ..th steps\n\
\n\
# Values specifying when to stop for INIT, otherwise same as above\n\
MaxInitMinStep\t100\t# Maximum number of steps\n\
InitRelEpsTotalE\t1e-05\t# relative change in total energy\n\
InitRelEpsKineticE\t0.0001\t# relative change in kinetic energy\n\
InitMaxMinStopStep\t2\t# check every ..th steps\n\
InitMaxMinGapStopStep\t1\t# check every ..th steps\n\
\n\
BoxLength\t# (Length of a unit cell)\n\
20\n\
0\t20\n\
0\t0\t20\n\
\n\
ECut\t128\t# energy cutoff for discretization in Hartrees\n\
MaxLevel\t5\t# number of different levels in the code, >=2\n\
Level0Factor\t2\t# factor by which node number increases from S to 0 level\n\
RiemannTensor\t0\t# (Use metric)\n\
PsiType\t0\t# 0 - doubly occupied, 1 - SpinUp,SpinDown\n\
MaxPsiDouble\t2\t# here: specifying both maximum number of SpinUp- and -Down-states\n\
PsiMaxNoUp\t2\t# here: specifying maximum number of SpinUp-states\n\
PsiMaxNoDown\t2\t# here: specifying maximum number of SpinDown-states\n\
AddPsis\t0\t# Additional unoccupied Psis for bandgap determination\n\
\n\
RCut\t20\t# R-cut for the ewald summation\n\
StructOpt\t0\t# Do structure optimization beforehand\n\
IsAngstroem\t1\t# 0 - Bohr, 1 - Angstroem\n\
RelativeCoord\t0\t# whether ion coordinates are relative (1) or absolute (0)\n\
MaxTypes\t2\t# maximum number of different ion types\n\
\n\
# Ion type data (PP = PseudoPotential, Z = atomic number)\n\
#Ion_TypeNr.\tAmount\tZ\tRGauss\tL_Max(PP)L_Loc(PP)IonMass\t# chemical name, symbol\n\
Ion_Type1\t2\t1\t1.0\t3\t3\t1.008\tHydrogen\tH\n\
Ion_Type2\t1\t8\t1.0\t3\t3\t15.999\tOxygen\tO\n\
#Ion_TypeNr._Nr.R[0]\tR[1]\tR[2]\tMoveType (0 MoveIon, 1 FixedIon)\n\
Ion_Type2_1\t0.000000000\t0.000000000\t0.000000000\t0 # molecule nr 0\n\
Ion_Type1_1\t0.758602\t0.000000000\t0.504284\t0 # molecule nr 1\n\
Ion_Type1_2\t0.758602\t0.000000000\t-0.504284\t0 # molecule nr 2\n";
static string waterMpqc ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<MBPT2>: (\n\
\t\tmaxiter = 200\n\
\t\tbasis = $:basis\n\
\t\tmolecule = $:molecule\n\
\t\treference<CLHF>: (\n\
\t\t\tbasis = $:basis\n\
\t\t\tmolecule = $:molecule\n\
\t\t)\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n";
static string waterXyz = "3\n\tH2O: water molecule\nO\t0\t0\t0\nH\t0.758602\t0\t0.504284\nH\t0.758602\t0\t-0.504284\n";
static string Tremolo_Atomdata1 = "# ATOMDATA\tId\tname\tType\tx=3\n";
static string Tremolo_Atomdata2 = "#\n#ATOMDATA Id name Type x=3\n1 hydrogen H 3.0 4.5 0.1\n\n";
static string Tremolo_invalidkey = "#\n#ATOMDATA Id name foo Type x=3\n\n\n";
static string Tremolo_velocity = "#\n#ATOMDATA Id name Type u=3\n1 hydrogen H 3.0 4.5 0.1\n\n";
static string Tremolo_neighbours = "#\n#ATOMDATA Id Type neighbors=2\n1 H 3 0\n2 H 3 0\n3 O 1 2\n";
static string Tremolo_improper = "#\n#ATOMDATA Id Type imprData\n8 H 9-10\n9 H 10-8,8-10\n10 O -\n";
static string Tremolo_torsion = "#\n#ATOMDATA Id Type torsion\n8 H 9-10\n9 H 10-8,8-10\n10 O -\n";
static string Tremolo_full = "# ATOMDATA\tx=3\tu=3\tF\tstress\tId\tneighbors=5\timprData\tGroupMeasureTypeNo\tType\textType\tname\tresName\tchainID\tresSeq\toccupancy\ttempFactor\tsegID\tCharge\tcharge\tGrpTypeNo\ttorsion\n0\t0\t0\t0\t0\t0\t0\t0\t0\t0\t0\t0\t0\t0\t-\t0\tH\t-\t-\t-\t0\t0\t0\t0\t0\t0\t0\t0\t-\t\n";

void ParserUnitTest::setUp() {
  World::getInstance();

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserUnitTest::tearDown() {
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserUnitTest::rewriteAnXyzTest() {
  cout << "Testing the XYZ parser." << endl;
  XyzParser* testParser = new XyzParser();
  stringstream input;
  input << waterXyz;
  testParser->load(&input);

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  string newWaterXyz = "";
  stringstream output;
  testParser->save(&output);
  newWaterXyz = output.str();

  CPPUNIT_ASSERT(waterXyz == newWaterXyz);
}

void ParserUnitTest::readTremoloPreliminaryCommentsTest() {
  cout << "Testing the tremolo parser." << endl;
  TremoloParser* testParser = new TremoloParser();
  stringstream input, output;

  // Atomdata beginning with "# ATOMDATA"
  input << Tremolo_Atomdata1;
  testParser->load(&input);
  testParser->save(&output);
  CPPUNIT_ASSERT(Tremolo_Atomdata1 == output.str());
  input.clear();
  output.clear();

  // Atomdata beginning with "#ATOMDATA"
  input << Tremolo_Atomdata2;
  testParser->load(&input);
  testParser->save(&output);
  CPPUNIT_ASSERT(output.str().find("hydrogen") != string::npos);
  input.clear();
  output.clear();

  // Invalid key in Atomdata line
  input << Tremolo_invalidkey;
  testParser->load(&input);
  //TODO: proove invalidity
  input.clear();
}

void ParserUnitTest::readTremoloCoordinatesTest() {
  TremoloParser* testParser = new TremoloParser();
  stringstream input;

  // One simple data line
  input << Tremolo_Atomdata2;
  testParser->load(&input);
  CPPUNIT_ASSERT(World::getInstance().getAtom(AtomByType(1))->at(0) == 3.0);
  input.clear();
}

void ParserUnitTest::readTremoloVelocityTest() {
  TremoloParser* testParser = new TremoloParser();
  stringstream input;

  // One simple data line
  input << Tremolo_velocity;
  testParser->load(&input);
  CPPUNIT_ASSERT(World::getInstance().getAtom(AtomByType(1))->AtomicVelocity[0] == 3.0);
  input.clear();
}

void ParserUnitTest::readTremoloNeighborInformationTest() {
  TremoloParser* testParser = new TremoloParser();
  stringstream input;

  // Neighbor data
  input << Tremolo_neighbours;
  testParser->load(&input);

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
  CPPUNIT_ASSERT(World::getInstance().getAtom(AtomByType(8))->
      IsBondedTo(World::getInstance().getAtom(AtomByType(1))));
  input.clear();
}

void ParserUnitTest::readAndWriteTremoloImprDataInformationTest() {
  TremoloParser* testParser = new TremoloParser();
  stringstream input, output;

  // Neighbor data
  input << Tremolo_improper;
  testParser->load(&input);
  testParser->save(&output);
  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
  CPPUNIT_ASSERT(output.str().find("2-0,0-2") != string::npos);
  input.clear();
  output.clear();
}

void ParserUnitTest::readAndWriteTremoloTorsionInformationTest() {
  TremoloParser* testParser = new TremoloParser();
  stringstream input, output;

  // Neighbor data
  input << Tremolo_torsion;
  testParser->load(&input);
  testParser->save(&output);
  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
  CPPUNIT_ASSERT(output.str().find("2-0,0-2") != string::npos);
  input.clear();
  output.clear();
}

void ParserUnitTest::writeTremoloTest() {
  TremoloParser* testParser = new TremoloParser();
  stringstream output;

  // with the maximum number of fields and minimal information, default values are printed
  atom* newAtom = World::getInstance().createAtom();
  newAtom->setType(1);
  testParser->setFieldsForSave("x=3 u=3 F stress Id neighbors=5 imprData GroupMeasureTypeNo Type extType name resName chainID resSeq occupancy tempFactor segID Charge charge GrpTypeNo torsion");
  testParser->save(&output);
  CPPUNIT_ASSERT(output.str() == Tremolo_full);

  cout << "testing the tremolo parser is done" << endl;
}

void ParserUnitTest::readwritePcpTest() {
  stringstream input(waterPcp);
  PcpParser* testParser = new PcpParser();
  testParser->load(&input);
  input.clear();

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  string newWaterPcp = "";
  stringstream output;
  testParser->save(&output);

  input << output;
  PcpParser* testParser2 = new PcpParser();
  testParser2->load(&input);

  CPPUNIT_ASSERT_EQUAL(6, World::getInstance().numAtoms());

  CPPUNIT_ASSERT(*testParser == *testParser2);
}

void ParserUnitTest::writeMpqcTest() {
  // build up water molecule
  atom *Walker = NULL;
  Walker = World::getInstance().createAtom();
  Walker->setType(8);
  Walker->setPosition(Vector(0,0,0));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,0.504284));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,-0.504284));
  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  // create two stringstreams, one stored, one created
  stringstream input(waterMpqc);
  MpqcParser* testParser = new MpqcParser();
  stringstream output;
  testParser->save(&output);

  // compare both configs
  string first = input.str();
  string second = output.str();
  CPPUNIT_ASSERT(first == second);
}
