/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * TypeEnumContainer.cpp
 *
 *  Created on: Oct 27, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/filesystem.hpp>
#include <iosfwd>
#include <typeinfo>
#include <vector>

#include "Actions/Values.hpp"
#include "CodePatterns/Assert.hpp"

#include "UIElements/CommandLineUI/TypeEnumContainer.hpp"

#include "atom.hpp"
#include "Box.hpp"
#include "LinearAlgebra/BoxVector.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "RandomNumbers/RandomNumberDistribution_Parameters.hpp"

/** Constructor for class TypeEnumContainer.
 * Just fills TypeEnumContainer::TypeEnumMap by hand.
 */
TypeEnumContainer::TypeEnumContainer()
{
  TypeEnumMap[&typeid(void)] = NoneType;
  TypeEnumMap[&typeid(bool)] = BooleanType;
  TypeEnumMap[&typeid(Box)] = BoxType;
  TypeEnumMap[&typeid(BoxVector)] = VectorType;
  TypeEnumMap[&typeid(Vector)] = VectorType;
  TypeEnumMap[&typeid(boost::filesystem::path)] = FileType;
  TypeEnumMap[&typeid(std::vector<boost::filesystem::path>)] = ListOfFilesType;
  TypeEnumMap[&typeid(int)] = IntegerType;
  TypeEnumMap[&typeid(std::vector<int>)] = ListOfIntegersType;
  TypeEnumMap[&typeid(unsigned int)] = UnsignedIntegerType;
  TypeEnumMap[&typeid(std::vector<unsigned int>)] = ListOfUnsignedIntegersType;
  TypeEnumMap[&typeid(double)] = DoubleType;
  TypeEnumMap[&typeid(std::vector<double>)] = ListOfDoublesType;
  TypeEnumMap[&typeid(std::string)] = StringType;
  TypeEnumMap[&typeid(std::vector<std::string>)] = ListOfStringsType;
  TypeEnumMap[&typeid(VectorValue)] = VectorType;
  TypeEnumMap[&typeid(std::vector<VectorValue>)] = ListOfVectorsType;
  TypeEnumMap[&typeid(const atom *)] = AtomType;
  TypeEnumMap[&typeid(std::vector<const atom *>)] = ListOfAtomsType;
  TypeEnumMap[&typeid(const molecule *)] = MoleculeType;
  TypeEnumMap[&typeid(std::vector<const molecule *>)] = ListOfMoleculesType;
  TypeEnumMap[&typeid(const element *)] = ElementType;
  TypeEnumMap[&typeid(std::vector<const element *>)] = ListOfElementsType;
  TypeEnumMap[&typeid(RandomNumberDistribution_Parameters)] = RandomNumberDistribution_ParametersType;

  // for debugging: list all stored types
  //ListAllKnownTypes();
};

/** Destructor for class TypeEnumContainer.
 * Clears TypeEnumContainer::TypeEnumMap.
 */
TypeEnumContainer::~TypeEnumContainer()
{
  TypeEnumMap.clear();
}

/** Destructor for class TypeEnumContainer.
 * Clears TypeEnumContainer::TypeEnumMap.
 */
void TypeEnumContainer::ListAllKnownTypes() const
{
  for (type_map::const_iterator iter = TypeEnumMap.begin(); iter != TypeEnumMap.end(); ++iter) {
    std::cout << " Known type is " << (iter->first)->name() << " or enum " << iter->second << std::endl;
  }
}

/** Getter for TypeEnumContainer::TypeEnumMap.
 * Note: \a *_type must be present, is checked by ASSERT().
 * \param *_type type requested
 */
enum TypeEnumContainer::EnumOfTypes TypeEnumContainer::getEnumforType(const std::type_info *_type)
{
  //std::cout << "Looking for type " << _type->name() << std::endl;
  ASSERT(TypeEnumMap.find(_type) != TypeEnumMap.end(),
      "CommandLineParser::getEnumforType() - cannot find the type as enum!.");
  return (TypeEnumMap[_type]);
}

