/*
 * Process.hpp
 *
 *  Created on: Feb 17, 2010
 *      Author: crueger
 */


/**
 * A Process is an action that might take some time and therfore contains methods
 * that allows showing how much is done.
 */
#ifndef PROCESS_HPP_
#define PROCESS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <set>

#include "CodePatterns/Observer/Observable.hpp"
#include "CodePatterns/Observer/Observer.hpp"
#include "Actions/Action.hpp"

namespace MoleCuilder {

class ActionRegistry;

/**
 * A Process is an Action that might take some time and therefore has special
 * methods to allow communication with progress indicators. Indicators
 * can sign on at a global place and will be notified when any process is doing
 * a calculation.
 *
 * A Process has four states:
 *  - starting: It is in the process of setting itself up, and wants everybody to know that it will start
 *              the calculation soon. Indicators should set up anything they need for displaying the progress
 *              when they are notified by a process in this state.
 *  - active:   The process is currently doing it's thing and wants any indicator to know it's status, i.e.
 *              the percentage done.
 *  - stopping: The process has fullfilled it's purpose and is shutting down. Indicators recieving this status
 *              should also use it to shut down their indication mechanism and delete any objects allocated for
 *              this Process
 *  - inactive: This Process is currently sleeping. If a Process is sending out any signals in this state, there
 *              is something seriously wrong.
 */
class Process : public Action, public Observable
{
public:
  Process(int _maxSteps, const ActionTrait &_trait);
  virtual ~Process();

  bool  isRunning();
  bool  doesStart();
  bool  doesStop();
  int   getCurrStep();
  void  setCurrStep(int _currStep);
  float getDoneRatio();
  int   getMaxSteps();
  void  setMaxSteps(int _maxSteps);

protected:
  void start();
  void step();
  void stop();

private:
  int currStep;
  int maxSteps;
  bool active;
  bool starts;
  bool stops;

  // some global static stuff to allow general Observers that can show progresses
public:
  static void AddObserver(Observer *);
  static void RemoveObserver(Observer *);
private:
  static std::set<Observer*> processObservers;
};

}

#endif /* PROCESS_HPP_ */
