/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Process.cpp
 *
 *  Created on: Feb 17, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <set>

#include "Process.hpp"

using namespace MoleCuilder;

Process::Process(int _maxSteps, const ActionTrait &_trait) :
  Action(_trait),
  Observable("Process"),
  maxSteps(_maxSteps),
  active(false),
  starts(false),
  stops(false)
{}

Process::~Process()
{
  // make sure everybody knows we have stoped
  stop();
}


bool Process::isRunning(){
  return active;
}

bool Process::doesStart(){
  return starts;
}

bool Process::doesStop(){
  return stops;
}

int Process::getCurrStep(){
  return currStep;
}

void Process::setCurrStep(int _currStep){
  OBSERVE;
  currStep = _currStep;
}

float Process::getDoneRatio() {
  if(getMaxSteps())
    return ((float)getCurrStep()/(float)getMaxSteps())*100.0;
  else
    return 0;
}
int Process::getMaxSteps(){
  return maxSteps;
}

void Process::setMaxSteps(int _maxSteps){
  maxSteps = _maxSteps;
}

void Process::start(){
  starts = true;
  {
    // we forcibly sign on all observers
    std::set<Observer*>::iterator iter;
    for(iter=processObservers.begin();iter!=processObservers.end();++iter){
      this->signOn((*iter));
    }
  }
  // only this small part should be observed
  {
    OBSERVE;
    currStep=0;
  }
  starts = false;
  active = true;
}

void Process::step(){
  OBSERVE;
  currStep++;
}

void Process::stop(){
  active=false;
  stops = true;
  {
    OBSERVE;
    currStep=0;
  }
  {
    // when we are done we forcibly sign off all observers
    std::set<Observer*>::iterator iter;
    for(iter=processObservers.begin();iter!=processObservers.end();++iter){
      this->signOff((*iter));
    }
  }
  stops = false;
}

// static stuff

std::set<Observer*> Process::processObservers;

void Process::AddObserver(Observer *obs){
  processObservers.insert(obs);
}
void Process::RemoveObserver(Observer *obs){
  processObservers.erase(obs);
}

