/*
 * ActionQueue.hpp
 *
 *  Created on: Aug 16, 2013
 *      Author: heber
 */

#ifndef ACTIONQUEUE_HPP_
#define ACTIONQUEUE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Singleton.hpp"

#include <list>
#include <vector>

#include "Actions/ActionState.hpp"

namespace MoleCuilder {

class Action;
class ActionHistory;
class ActionRegistry;
class ActionTrait;

/** This class combines the whole handling of Actions into a single class.
 *
 * It spawns new Actions by its internal ActionRegistry. Spawned Actions are
 * automatically queued and placed into a History after execution.
 */
class ActionQueue : public Singleton<ActionQueue>
{
  friend class Singleton<ActionQueue>;
public:
  typedef std::vector<std::string> ActionTokens_t;
  typedef std::list<Action *> ActionQueue_t;

  /** Returns the spawned action by token \a name.
   *
   * Action is checked into internal queue.
   *
   * \return pointer to newly spawned action
   */
  Action* getActionByName(const std::string &name);

  /** Checks whether the Action is known by the given \a name.
   *
   * \param name token of action to check
   * \return true - token is known, false - else
   */
  bool isActionKnownByName(const std::string &name) const;

  /** Register an Action with the ActionRegistry.
   *
   * \param _action action to add
   */
  void registerAction(Action *_action);

  /** Returns the vector with the tokens of all currently known Actions.
   *
   * \return list of all tokens
   */
  const ActionTokens_t getListOfActions() const;

  /** Returns the trait to an Action.
   *
   * \param name name of Action
   * \return const ref to its default Trait.
   */
  const ActionTrait& getActionsTrait(const std::string &name) const;

  /** Undoes last called Action.
   *
   */
  void undoLast();

  /** Redoes last undone Action.
   *
   */
  void redoLast();

private:
  //!> grant Action access to internal history functions.
  friend class Action;

  /** Wrapper function to add state to ActionHistory.
   *
   * \param _Action Action whose state to add
   * \param _state state to add
   */
  void addElement(Action* _Action, ActionState::ptr _state);

  /** Wrapper function to clear ActionHistory.
   *
   */
  void clear();

private:
  /** Private cstor for ActionQueue.
   *
   * Must be private as is singleton.
   *
   */
  ActionQueue();

  /** Dstor for ActionQueue.
   *
   */
  ~ActionQueue();

  //!> ActionRegistry to spawn new actions
  ActionRegistry *AR;

  //!> ActionHistory is for undoing and redoing actions, requires ActionRegistry fully initialized
  ActionHistory *history;

  //!> internal queue of actions
  ActionQueue_t queue;
};

};

#endif /* ACTIONQUEUE_HPP_ */
