/*
 * HydrogenPool.hpp
 *
 *  Created on: Mar 3, 2013
 *      Author: heber
 */

#ifndef HYDROGENPOOL_HPP_
#define HYDROGENPOOL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <deque>
#include <map>

#include "types.hpp"

class atom;

/** This class contains a pool of hydrogen meant for saturation.
 *
 * These hydrogen atoms do not trigger any updates, they are mostly
 * invisible to the system.
 *
 * If the pool is lacking hydrogens, it requests more from the World.
 *
 * The idea is that we always create a single fragment, saturate it with
 * hydrogen atoms from the pool, then store it to file or convert to a
 * job. Directly afterwards, the fragment is free'd and the hydrogen may
 * go back into the pool.
 */
class HydrogenPool
{
  //!> grant unit test access to private parts
  friend class HydrogenPoolTest;
public:
  HydrogenPool();
  ~HydrogenPool();

  /** Delivers a hydrogen atom for use.
   *
   * \return point to the hydrogen atom
   */
  atom * leaseHydrogen();

  /** Releases a used hydrogen again.
   *
   * \param _atom hydrogen atom to release
   */
  void releaseHydrogen(atom * _atom);

  /** Releases a used hydrogen again.
   *
   * \param _atom hydrogen atom to release
   */
  void releaseHydrogen(atomId_t _atom);

  //!> enum to define the atomic number of atoms contained in the pool
  enum { HYDROGEN = 1 };

private:
  /** Requests a hydrogen atom from the World and places it into the queue.
   *
   */
  void requestHydrogenIntoPool();

  /** Helper function that removes all present hydrogen atoms again.
   *
   */
  void cleanup();

  /** Helper function to make sure \a _atom is up to current time step.
   *
   * \param _atom atom to bring trajectory size up to speed
   */
  void UpdateSteps(atom * _atom) const;


private:
  //!> typedef for the deque of available hydrogens.
  typedef std::deque<atom *> HydrogenQueue_t;
  //!> typedef for the map of all hydrogens currently in use in a fragment.
  typedef std::map<atomId_t, atom *> HydrogenInUse_t;

  //!> enum to define a warning constant for the limit of hydrogens
  enum { WARNINGTHRESHOLD = 100 };

  //!> counter for the total number of hydrogens requested so far.
  int HydrogenCount;
  //!> Queue of available hydrogen atoms
  HydrogenQueue_t HydrogenQueue;
  //!> Map for looking up ids of hydrogen in use for reinsertion into queue
  HydrogenInUse_t HydrogenInUse;
};

#endif /* HYDROGENPOOL_HPP_ */
