/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * controller_MPQCCommandJob.cpp
 *
 *  Created on: 01.06.2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "controller_MPQCCommandJob.hpp"

#include <boost/assign.hpp>
#include <boost/bind.hpp>
#include <fstream>

#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"

#include "JobMarket/Controller/ControllerCommand.hpp"
#include "JobMarket/Controller/ControllerCommandRegistry.hpp"
#include "JobMarket/Controller/FragmentController.hpp"
#include "JobMarket/Jobs/FragmentJob.hpp"
#include "JobMarket/Results/FragmentResult.hpp"

#include "Fragmentation/EnergyMatrix.hpp"
#include "Fragmentation/ForceMatrix.hpp"
#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/defs.hpp"

#include "Helpers/defs.hpp"

#include "Jobs/MPQCCommandJob.hpp"
#include "Jobs/MPQCCommandJob_MPQCData.hpp"

#include "ControllerOptions_MPQCCommandJob.hpp"

/** Creates a MPQCCommandJob with argument \a filename.
 *
 * @param jobs created job is added to this vector
 * @param command mpqc command to execute
 * @param filename filename being argument to job
 * @param nextid id for this job
 */
void parsejob(
    std::vector<FragmentJob::ptr> &jobs,
    const std::string &command,
    const std::string &filename,
    const JobId_t nextid)
{
  std::ifstream file;
  file.open(filename.c_str());
  ASSERT( file.good(), "parsejob() - file "+filename+" does not exist.");
  std::string output((std::istreambuf_iterator<char>(file)),
      std::istreambuf_iterator<char>());
  FragmentJob::ptr testJob( new MPQCCommandJob(output, nextid, command) );
  jobs.push_back(testJob);
  file.close();
  LOG(1, "INFO: Added MPQCCommandJob from file "+filename+".");
}

/** Print received results.
 *
 * @param results received results to print
 */
void printReceivedResults(const std::vector<FragmentResult::ptr> &results)
{
  for (std::vector<FragmentResult::ptr>::const_iterator iter = results.begin();
      iter != results.end(); ++iter)
    LOG(1, "RESULT: job #"+toString((*iter)->getId())+": "+toString((*iter)->result));
}

/** Print MPQCData from received results.
 *
 * @param results received results to extract MPQCData from
 * @param KeySetFilename filename with keysets to associate forces correctly
 * @param NoAtoms total number of atoms
 */
bool printReceivedMPQCResults(
    const std::vector<FragmentResult::ptr> &results,
    const std::string &KeySetFilename,
    size_t NoAtoms)
{
  EnergyMatrix Energy;
  EnergyMatrix EnergyFragments;
  ForceMatrix Force;
  ForceMatrix ForceFragments;
  KeySetsContainer KeySet;

  // align fragments
  std::map< JobId_t, size_t > MatrixNrLookup;
  size_t FragmentCounter = 0;
  {
    // bring ids in order ...
    typedef std::map< JobId_t, FragmentResult::ptr> IdResultMap_t;
    IdResultMap_t IdResultMap;
    for (std::vector<FragmentResult::ptr>::const_iterator iter = results.begin();
        iter != results.end(); ++iter) {
  #ifndef NDEBUG
      std::pair< IdResultMap_t::iterator, bool> inserter =
  #endif
      IdResultMap.insert( make_pair((*iter)->getId(), *iter) );
      ASSERT( inserter.second,
          "printReceivedMPQCResults() - two results have same id "
          +toString((*iter)->getId())+".");
    }
    // ... and fill lookup
    for(IdResultMap_t::const_iterator iter = IdResultMap.begin();
        iter != IdResultMap.end(); ++iter)
      MatrixNrLookup.insert( make_pair(iter->first, FragmentCounter++) );
  }
  LOG(1, "INFO: There are " << FragmentCounter << " fragments.");

  // extract results
  std::vector<MPQCData> fragmentData(results.size());
  MPQCData combinedData;

  LOG(2, "DEBUG: Parsing now through " << results.size() << " results.");
  for (std::vector<FragmentResult::ptr>::const_iterator iter = results.begin();
      iter != results.end(); ++iter) {
    LOG(1, "RESULT: job #"+toString((*iter)->getId())+": "+toString((*iter)->result));
    MPQCData extractedData;
    std::stringstream inputstream((*iter)->result);
    LOG(2, "DEBUG: First 50 characters FragmentResult's string: "+(*iter)->result.substr(0, 50));
    boost::archive::text_iarchive ia(inputstream);
    ia >> extractedData;
    LOG(1, "INFO: extracted data is " << extractedData << ".");

    // place results into EnergyMatrix ...
    {
      MatrixContainer::MatrixArray matrix;
      matrix.resize(1);
      matrix[0].resize(1, extractedData.energy);
      if (!Energy.AddMatrix(
          std::string("MPQCJob ")+toString((*iter)->getId()),
          matrix,
          MatrixNrLookup[(*iter)->getId()])) {
        ELOG(1, "Adding energy matrix failed.");
        return false;
      }
    }
    // ... and ForceMatrix (with two empty columns in front)
    {
      MatrixContainer::MatrixArray matrix;
      const size_t rows = extractedData.forces.size();
      matrix.resize(rows);
      for (size_t i=0;i<rows;++i) {
        const size_t columns = 2+extractedData.forces[i].size();
        matrix[i].resize(columns, 0.);
  //      for (size_t j=0;j<2;++j)
  //        matrix[i][j] = 0.;
        for (size_t j=2;j<columns;++j)
          matrix[i][j] = extractedData.forces[i][j-2];
      }
      if (!Force.AddMatrix(
          std::string("MPQCJob ")+toString((*iter)->getId()),
          matrix,
          MatrixNrLookup[(*iter)->getId()])) {
        ELOG(1, "Adding force matrix failed.");
        return false;
      }
    }
  }
  // add one more matrix (not required for energy)
  MatrixContainer::MatrixArray matrix;
  matrix.resize(1);
  matrix[0].resize(1, 0.);
  if (!Energy.AddMatrix(std::string("MPQCJob total"), matrix, FragmentCounter))
    return false;
  // but for energy because we need to know total number of atoms
  matrix.resize(NoAtoms);
  for (size_t i = 0; i< NoAtoms; ++i)
    matrix[i].resize(2+NDIM, 0.);
  if (!Force.AddMatrix(std::string("MPQCJob total"), matrix, FragmentCounter))
    return false;


  // combine all found data
  if (!Energy.InitialiseIndices()) return false;

  if (!Force.ParseIndices(KeySetFilename.c_str())) return false;

  if (!KeySet.ParseKeySets(KeySetFilename.c_str(), Force.RowCounter, Force.MatrixCounter)) return false;

  if (!KeySet.ParseManyBodyTerms()) return false;

  if (!EnergyFragments.AllocateMatrix(Energy.Header, Energy.MatrixCounter, Energy.RowCounter, Energy.ColumnCounter)) return false;
  if (!ForceFragments.AllocateMatrix(Force.Header, Force.MatrixCounter, Force.RowCounter, Force.ColumnCounter)) return false;

  if(!Energy.SetLastMatrix(0., 0)) return false;
  if(!Force.SetLastMatrix(0., 2)) return false;

  for (int BondOrder=0;BondOrder<KeySet.Order;BondOrder++) {
    // --------- sum up energy --------------------
    LOG(1, "INFO: Summing energy of order " << BondOrder+1 << " ...");
    if (!EnergyFragments.SumSubManyBodyTerms(Energy, KeySet, BondOrder)) return false;
    if (!Energy.SumSubEnergy(EnergyFragments, NULL, KeySet, BondOrder, 1.)) return false;

    // --------- sum up Forces --------------------
    LOG(1, "INFO: Summing forces of order " << BondOrder+1 << " ...");
    if (!ForceFragments.SumSubManyBodyTerms(Force, KeySet, BondOrder)) return false;
    if (!Force.SumSubForces(ForceFragments, KeySet, BondOrder, 1.)) return false;
  }

  // for debugging print resulting energy and forces
  LOG(1, "INFO: Resulting energy is " << Energy.Matrix[ FragmentCounter ][0][0]);
  std::stringstream output;
  for (int i=0; i< Force.RowCounter[FragmentCounter]; ++i) {
    for (int j=0; j< Force.ColumnCounter[FragmentCounter]; ++j)
      output << Force.Matrix[ FragmentCounter ][i][j] << " ";
    output << "\n";
  }
  LOG(1, "INFO: Resulting forces are " << std::endl << output.str());

  return true;
}


/** Helper function to get number of atoms somehow.
 *
 * Here, we just parse the number of lines in the adjacency file as
 * it should correspond to the number of atoms, except when some atoms
 * are not bonded, but then fragmentation makes no sense.
 *
 * @param path path to the adjacency file
 */
size_t getNoAtomsFromAdjacencyFile(const std::string &path)
{
  size_t NoAtoms = 0;

  // parse in special file to get atom count (from line count)
  std::string filename(path);
  filename += FRAGMENTPREFIX;
  filename += ADJACENCYFILE;
  std::ifstream adjacency(filename.c_str());
  if (adjacency.fail()) {
    LOG(0, endl << "getNoAtomsFromAdjacencyFile() - Unable to open " << filename << ", is the directory correct?");
    return false;
  }
  std::string buffer;
  while (getline(adjacency, buffer))
    NoAtoms++;
  LOG(1, "INFO: There are " << NoAtoms << " atoms.");

  return NoAtoms;
}

/** Creates a MPQCCommandJob out of give \a command with \a argument.
 *
 * @param controller reference to controller to add jobs
 * @param ControllerInfo information on the job
 */
void AddJobs(FragmentController &controller, const ControllerOptions_MPQCCommandJob &ControllerInfo)
{
  std::vector<FragmentJob::ptr> jobs;
  for (std::vector< std::string >::const_iterator iter = ControllerInfo.jobfiles.begin();
      iter != ControllerInfo.jobfiles.end(); ++iter) {
    const JobId_t next_id = controller.getAvailableId();
    const std::string &filename = *iter;
    LOG(1, "INFO: Creating MPQCCommandJob with filename '"
        +filename+"', and id "+toString(next_id)+".");
    parsejob(jobs, ControllerInfo.executable, filename, next_id);
  }
  controller.addJobs(jobs);
  controller.sendJobs(ControllerInfo.server, ControllerInfo.serverport);
}

inline std::vector<std::string> getListOfCommands(const ControllerCommandRegistry &ControllerCommands)
{
  std::vector<std::string> Commands;
  for (ControllerCommandRegistry::const_iterator iter = ControllerCommands.getBeginIter();
      iter != ControllerCommands.getEndIter(); ++iter)
    Commands.push_back(iter->first);

  return Commands;
}

ControllerOptions *controller_MPQCCommandJob::allocateControllerInfo()
{
  return new ControllerOptions_MPQCCommandJob();
}

void controller_MPQCCommandJob::addSpecificCommands(
    boost::function<void (ControllerCommand *)> &registrator,
    FragmentController &controller,
    ControllerOptions &ControllerInfo)
{
  ControllerOptions_MPQCCommandJob &CI =
      reinterpret_cast<ControllerOptions_MPQCCommandJob &>(ControllerInfo);
  registrator(new ControllerCommand("addjobs",
      boost::assign::list_of< ControllerCommand::commands_t >
        (boost::bind(&FragmentController::requestIds,
            boost::ref(controller), boost::cref(ControllerInfo.server), boost::cref(ControllerInfo.serverport),
            boost::bind(&std::vector<std::string>::size, boost::cref(CI.jobfiles))))
        (boost::bind(&AddJobs, boost::ref(controller), boost::cref(CI)))
      ));
  registrator(new ControllerCommand("receiveresults",
      boost::assign::list_of< ControllerCommand::commands_t >
        (boost::bind(&FragmentController::receiveResults,
            boost::ref(controller), boost::cref(ControllerInfo.server), boost::cref(ControllerInfo.serverport)))
        (boost::bind(&printReceivedResults,
            boost::bind(&FragmentController::getReceivedResults, boost::ref(controller))))
      ));
  registrator(new ControllerCommand("receivempqc",
      boost::assign::list_of< ControllerCommand::commands_t >
        (boost::bind(&FragmentController::receiveResults,
            boost::ref(controller), boost::cref(ControllerInfo.server), boost::cref(ControllerInfo.serverport)))
        (boost::bind(&printReceivedMPQCResults,
            boost::bind(&FragmentController::getReceivedResults, boost::ref(controller)),
            boost::cref(CI.fragmentpath),
            boost::bind(&getNoAtomsFromAdjacencyFile, boost::cref(CI.fragmentpath))))
  ));
}

void controller_MPQCCommandJob::addSpecificOptions(
    boost::program_options::options_description_easy_init option)
{
  option
    ("executable", boost::program_options::value< std::string >(), "executable for commands 'createjobs'")
    ("fragment-path", boost::program_options::value< std::string >(), "path to fragment files for 'receivempqc'")
    ("jobfiles", boost::program_options::value< std::vector< std::string > >()->multitoken(), "list of files as single argument toexecutable for 'addjobs'")
    ;
}

int controller_MPQCCommandJob::addOtherParsings(
    ControllerOptions &ControllerInfo,
    boost::program_options::variables_map &vm)
{
  ControllerOptions_MPQCCommandJob &CI =
      reinterpret_cast<ControllerOptions_MPQCCommandJob &>(ControllerInfo);
  int status = 0;
  status = CI.parseExecutable(vm);
  if (status) return status;
  status = CI.parseFragmentpath(vm);
  if (status) return status;
  status = CI.parseJobfiles(vm);
  return status;
}
