/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/** \file config.cpp
 *
 * Function implementations for the class config.
 *
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <stdio.h>
#include <cstring>

//#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "config.hpp"
#include "ConfigFileBuffer.hpp"
#include "element.hpp"
#include "Helpers/helpers.hpp"
#include "Helpers/Info.hpp"
#include "lists.hpp"
#include "Helpers/Verbose.hpp"
#include "Helpers/Log.hpp"
#include "molecule.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "ThermoStatContainer.hpp"
#include "World.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "Box.hpp"

/************************************* Functions for class config ***************************/

/** Constructor for config file class.
 */
config::config() :
  BG(NULL),
  Thermostats(0),
  PsiType(0),
  MaxPsiDouble(0),
  PsiMaxNoUp(0),
  PsiMaxNoDown(0),
  MaxMinStopStep(1),
  InitMaxMinStopStep(1),
  ProcPEGamma(8),
  ProcPEPsi(1),
  configname(NULL),
  FastParsing(false),
  Deltat(0.01),
  basis(""),
  databasepath(NULL),
  DoConstrainedMD(0),
  MaxOuterStep(0),
  mainname(NULL),
  defaultpath(NULL),
  pseudopotpath(NULL),
  DoOutVis(0),
  DoOutMes(1),
  DoOutNICS(0),
  DoOutOrbitals(0),
  DoOutCurrent(0),
  DoFullCurrent(0),
  DoPerturbation(0),
  DoWannier(0),
  CommonWannier(0),
  SawtoothStart(0.01),
  VectorPlane(0),
  VectorCut(0.),
  UseAddGramSch(1),
  Seed(1),
  OutVisStep(10),
  OutSrcStep(5),
  MaxPsiStep(0),
  EpsWannier(1e-7),
  MaxMinStep(100),
  RelEpsTotalEnergy(1e-7),
  RelEpsKineticEnergy(1e-5),
  MaxMinGapStopStep(0),
  MaxInitMinStep(100),
  InitRelEpsTotalEnergy(1e-5),
  InitRelEpsKineticEnergy(1e-4),
  InitMaxMinGapStopStep(0),
  ECut(128.),
  MaxLevel(5),
  RiemannTensor(0),
  LevRFactor(0),
  RiemannLevel(0),
  Lev0Factor(2),
  RTActualUse(0),
  AddPsis(0),
  RCut(20.),
  StructOpt(0),
  IsAngstroem(1),
  RelativeCoord(0),
  MaxTypes(0)
{
  mainname = new char[MAXSTRINGSIZE];
  defaultpath = new char[MAXSTRINGSIZE];
  pseudopotpath = new char[MAXSTRINGSIZE];
  databasepath = new char[MAXSTRINGSIZE];
  configname = new char[MAXSTRINGSIZE];
  Thermostats = new ThermoStatContainer();
  strcpy(mainname,"pcp");
  strcpy(defaultpath,"not specified");
  strcpy(pseudopotpath,"not specified");
  configname[0]='\0';
  basis = "3-21G";
};

/** Destructor for config file class.
 */
config::~config()
{
  delete[](mainname);
  delete[](defaultpath);
  delete[](pseudopotpath);
  delete[](databasepath);
  delete[](configname);
  if (Thermostats != NULL)
    delete(Thermostats);

  if (BG != NULL)
    delete(BG);
};

/** Displays menu for editing each entry of the config file.
 * Nothing fancy here, just lots of Log() << Verbose(0)s for the menu and a switch/case
 * for each entry of the config file structure.
 */
void config::Edit()
{
  char choice;

  do {
    DoLog(0) && (Log() << Verbose(0) << "===========EDIT CONFIGURATION============================" << endl);
    DoLog(0) && (Log() << Verbose(0) << " A - mainname (prefix for all runtime files)" << endl);
    DoLog(0) && (Log() << Verbose(0) << " B - Default path (for runtime files)" << endl);
    DoLog(0) && (Log() << Verbose(0) << " C - Path of pseudopotential files" << endl);
    DoLog(0) && (Log() << Verbose(0) << " D - Number of coefficient sharing processes" << endl);
    DoLog(0) && (Log() << Verbose(0) << " E - Number of wave function sharing processes" << endl);
    DoLog(0) && (Log() << Verbose(0) << " F - 0: Don't output density for OpenDX, 1: do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " G - 0: Don't output physical data, 1: do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " H - 0: Don't output densities of each unperturbed orbital for OpenDX, 1: do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " I - 0: Don't output current density for OpenDX, 1: do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " J - 0: Don't do the full current calculation, 1: do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " K - 0: Don't do perturbation calculation to obtain susceptibility and shielding, 1: do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " L - 0: Wannier centres as calculated, 1: common centre for all, 2: unite centres according to spread, 3: cell centre, 4: shifted to nearest grid point" << endl);
    DoLog(0) && (Log() << Verbose(0) << " M - Absolute begin of unphysical sawtooth transfer for position operator within cell" << endl);
    DoLog(0) && (Log() << Verbose(0) << " N - (0,1,2) x,y,z-plane to do two-dimensional current vector cut" << endl);
    DoLog(0) && (Log() << Verbose(0) << " O - Absolute position along vector cut axis for cut plane" << endl);
    DoLog(0) && (Log() << Verbose(0) << " P - Additional Gram-Schmidt-Orthonormalization to stabilize numerics" << endl);
    DoLog(0) && (Log() << Verbose(0) << " Q - Initial integer value of random number generator" << endl);
    DoLog(0) && (Log() << Verbose(0) << " R - for perturbation 0, for structure optimization defines upper limit of iterations" << endl);
    DoLog(0) && (Log() << Verbose(0) << " T - Output visual after ...th step" << endl);
    DoLog(0) && (Log() << Verbose(0) << " U - Output source densities of wave functions after ...th step" << endl);
    DoLog(0) && (Log() << Verbose(0) << " X - minimization iterations per wave function, if unsure leave at default value 0" << endl);
    DoLog(0) && (Log() << Verbose(0) << " Y - tolerance value for total spread in iterative Jacobi diagonalization" << endl);
    DoLog(0) && (Log() << Verbose(0) << " Z - Maximum number of minimization iterations" << endl);
    DoLog(0) && (Log() << Verbose(0) << " a - Relative change in total energy to stop min. iteration" << endl);
    DoLog(0) && (Log() << Verbose(0) << " b - Relative change in kinetic energy to stop min. iteration" << endl);
    DoLog(0) && (Log() << Verbose(0) << " c - Check stop conditions every ..th step during min. iteration" << endl);
    DoLog(0) && (Log() << Verbose(0) << " e - Maximum number of minimization iterations during initial level" << endl);
    DoLog(0) && (Log() << Verbose(0) << " f - Relative change in total energy to stop min. iteration during initial level" << endl);
    DoLog(0) && (Log() << Verbose(0) << " g - Relative change in kinetic energy to stop min. iteration during initial level" << endl);
    DoLog(0) && (Log() << Verbose(0) << " h - Check stop conditions every ..th step during min. iteration during initial level" << endl);
//    Log() << Verbose(0) << " j - six lower diagonal entries of matrix, defining the unit cell" << endl;
    DoLog(0) && (Log() << Verbose(0) << " k - Energy cutoff of plane wave basis in Hartree" << endl);
    DoLog(0) && (Log() << Verbose(0) << " l - Maximum number of levels in multi-level-ansatz" << endl);
    DoLog(0) && (Log() << Verbose(0) << " m - Factor by which grid nodes increase between standard and upper level" << endl);
    DoLog(0) && (Log() << Verbose(0) << " n - 0: Don't use RiemannTensor, 1: Do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " o - Factor by which grid nodes increase between Riemann and standard(?) level" << endl);
    DoLog(0) && (Log() << Verbose(0) << " p - Number of Riemann levels" << endl);
    DoLog(0) && (Log() << Verbose(0) << " r - 0: Don't Use RiemannTensor, 1: Do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " s - 0: Doubly occupied orbitals, 1: Up-/Down-Orbitals" << endl);
    DoLog(0) && (Log() << Verbose(0) << " t - Number of orbitals (depends pn SpinType)" << endl);
    DoLog(0) && (Log() << Verbose(0) << " u - Number of SpinUp orbitals (depends on SpinType)" << endl);
    DoLog(0) && (Log() << Verbose(0) << " v - Number of SpinDown orbitals (depends on SpinType)" << endl);
    DoLog(0) && (Log() << Verbose(0) << " w - Number of additional, unoccupied orbitals" << endl);
    DoLog(0) && (Log() << Verbose(0) << " x - radial cutoff for ewald summation in Bohrradii" << endl);
    DoLog(0) && (Log() << Verbose(0) << " y - 0: Don't do structure optimization beforehand, 1: Do" << endl);
    DoLog(0) && (Log() << Verbose(0) << " z - 0: Units are in Bohr radii, 1: units are in Aengstrom" << endl);
    DoLog(0) && (Log() << Verbose(0) << " i - 0: Coordinates given in file are absolute, 1: ... are relative to unit cell" << endl);
    DoLog(0) && (Log() << Verbose(0) << "=========================================================" << endl);
    DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
    cin >> choice;

    switch (choice) {
        case 'A': // mainname
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::mainname << "\t new: ");
          cin >> config::mainname;
          break;
        case 'B': // defaultpath
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::defaultpath << "\t new: ");
          cin >> config::defaultpath;
          break;
        case 'C': // pseudopotpath
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::pseudopotpath << "\t new: ");
          cin >> config::pseudopotpath;
          break;

        case 'D': // ProcPEGamma
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::ProcPEGamma << "\t new: ");
          cin >> config::ProcPEGamma;
          break;
        case 'E': // ProcPEPsi
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::ProcPEPsi << "\t new: ");
          cin >> config::ProcPEPsi;
          break;
        case 'F': // DoOutVis
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::DoOutVis << "\t new: ");
          cin >> config::DoOutVis;
          break;
        case 'G': // DoOutMes
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::DoOutMes << "\t new: ");
          cin >> config::DoOutMes;
          break;
        case 'H': // DoOutOrbitals
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::DoOutOrbitals << "\t new: ");
          cin >> config::DoOutOrbitals;
          break;
        case 'I': // DoOutCurrent
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::DoOutCurrent << "\t new: ");
          cin >> config::DoOutCurrent;
          break;
        case 'J': // DoFullCurrent
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::DoFullCurrent << "\t new: ");
          cin >> config::DoFullCurrent;
          break;
        case 'K': // DoPerturbation
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::DoPerturbation << "\t new: ");
          cin >> config::DoPerturbation;
          break;
        case 'L': // CommonWannier
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::CommonWannier << "\t new: ");
          cin >> config::CommonWannier;
          break;
        case 'M': // SawtoothStart
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::SawtoothStart << "\t new: ");
          cin >> config::SawtoothStart;
          break;
        case 'N': // VectorPlane
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::VectorPlane << "\t new: ");
          cin >> config::VectorPlane;
          break;
        case 'O': // VectorCut
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::VectorCut << "\t new: ");
          cin >> config::VectorCut;
          break;
        case 'P': // UseAddGramSch
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::UseAddGramSch << "\t new: ");
          cin >> config::UseAddGramSch;
          break;
        case 'Q': // Seed
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::Seed << "\t new: ");
          cin >> config::Seed;
          break;

        case 'R': // MaxOuterStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxOuterStep << "\t new: ");
          cin >> config::MaxOuterStep;
          break;
        case 'T': // OutVisStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::OutVisStep << "\t new: ");
          cin >> config::OutVisStep;
          break;
        case 'U': // OutSrcStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::OutSrcStep << "\t new: ");
          cin >> config::OutSrcStep;
          break;
        case 'X': // MaxPsiStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxPsiStep << "\t new: ");
          cin >> config::MaxPsiStep;
          break;
        case 'Y': // EpsWannier
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::EpsWannier << "\t new: ");
          cin >> config::EpsWannier;
          break;

        case 'Z': // MaxMinStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxMinStep << "\t new: ");
          cin >> config::MaxMinStep;
          break;
        case 'a': // RelEpsTotalEnergy
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RelEpsTotalEnergy << "\t new: ");
          cin >> config::RelEpsTotalEnergy;
          break;
        case 'b': // RelEpsKineticEnergy
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RelEpsKineticEnergy << "\t new: ");
          cin >> config::RelEpsKineticEnergy;
          break;
        case 'c': // MaxMinStopStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxMinStopStep << "\t new: ");
          cin >> config::MaxMinStopStep;
          break;
        case 'e': // MaxInitMinStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxInitMinStep << "\t new: ");
          cin >> config::MaxInitMinStep;
          break;
        case 'f': // InitRelEpsTotalEnergy
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::InitRelEpsTotalEnergy << "\t new: ");
          cin >> config::InitRelEpsTotalEnergy;
          break;
        case 'g': // InitRelEpsKineticEnergy
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::InitRelEpsKineticEnergy << "\t new: ");
          cin >> config::InitRelEpsKineticEnergy;
          break;
        case 'h': // InitMaxMinStopStep
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::InitMaxMinStopStep << "\t new: ");
          cin >> config::InitMaxMinStopStep;
          break;

//        case 'j': // BoxLength
//          Log() << Verbose(0) << "enter lower triadiagonalo form of basis matrix" << endl << endl;
//          double * const cell_size = World::getInstance().getDomain();
//          for (int i=0;i<6;i++) {
//            Log() << Verbose(0) << "Cell size" << i << ": ";
//            cin >> cell_size[i];
//          }
//          break;

        case 'k': // ECut
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::ECut << "\t new: ");
          cin >> config::ECut;
          break;
        case 'l': // MaxLevel
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxLevel << "\t new: ");
          cin >> config::MaxLevel;
          break;
        case 'm': // RiemannTensor
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RiemannTensor << "\t new: ");
          cin >> config::RiemannTensor;
          break;
        case 'n': // LevRFactor
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::LevRFactor << "\t new: ");
          cin >> config::LevRFactor;
          break;
        case 'o': // RiemannLevel
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RiemannLevel << "\t new: ");
          cin >> config::RiemannLevel;
          break;
        case 'p': // Lev0Factor
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::Lev0Factor << "\t new: ");
          cin >> config::Lev0Factor;
          break;
        case 'r': // RTActualUse
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RTActualUse << "\t new: ");
          cin >> config::RTActualUse;
          break;
        case 's': // PsiType
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::PsiType << "\t new: ");
          cin >> config::PsiType;
          break;
        case 't': // MaxPsiDouble
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::MaxPsiDouble << "\t new: ");
          cin >> config::MaxPsiDouble;
          break;
        case 'u': // PsiMaxNoUp
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::PsiMaxNoUp << "\t new: ");
          cin >> config::PsiMaxNoUp;
          break;
        case 'v': // PsiMaxNoDown
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::PsiMaxNoDown << "\t new: ");
          cin >> config::PsiMaxNoDown;
          break;
        case 'w': // AddPsis
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::AddPsis << "\t new: ");
          cin >> config::AddPsis;
          break;

        case 'x': // RCut
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RCut << "\t new: ");
          cin >> config::RCut;
          break;
        case 'y': // StructOpt
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::StructOpt << "\t new: ");
          cin >> config::StructOpt;
          break;
        case 'z': // IsAngstroem
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::IsAngstroem << "\t new: ");
          cin >> config::IsAngstroem;
          break;
        case 'i': // RelativeCoord
          DoLog(0) && (Log() << Verbose(0) << "Old: " << config::RelativeCoord << "\t new: ");
          cin >> config::RelativeCoord;
          break;
    };
  } while (choice != 'q');
};

/** Tests whether a given configuration file adhears to old or new syntax.
 * \param *filename filename of config file to be tested
 * \param *periode pointer to a periodentafel class with all elements
 * \return 0 - old syntax, 1 - new syntax, -1 - unknown syntax
 */
int config::TestSyntax(const char * const filename, const periodentafel * const periode) const
{
  int test;
  ifstream file(filename);

  // search file for keyword: ProcPEGamma (new syntax)
  if (ParseForParameter(1,&file,"ProcPEGamma", 0, 1, 1, int_type, &test, 1, optional)) {
    file.close();
    return 1;
  }
  // search file for keyword: ProcsGammaPsi (old syntax)
  if (ParseForParameter(1,&file,"ProcsGammaPsi", 0, 1, 1, int_type, &test, 1, optional)) {
    file.close();
    return 0;
  }
  file.close();
  return -1;
}

/** Returns private config::IsAngstroem.
 * \return IsAngstroem
 */
bool config::GetIsAngstroem() const
{
  return (IsAngstroem == 1);
};

/** Returns private config::*defaultpath.
 * \return *defaultpath
 */
char * config::GetDefaultPath() const
{
  return defaultpath;
};


/** Returns private config::*defaultpath.
 * \return *defaultpath
 */
void config::SetDefaultPath(const char * const path)
{
  strcpy(defaultpath, path);
};

/** Loads a molecule from a ConfigFileBuffer.
 * \param *mol molecule to load
 * \param *FileBuffer ConfigFileBuffer to use
 * \param *periode periodentafel for finding elements
 * \param FastParsing whether to parse trajectories or not
 */
void LoadMolecule(molecule * const &mol, struct ConfigFileBuffer * const &FileBuffer, const periodentafel * const periode, const bool FastParsing)
{
  int MaxTypes = 0;
  const element *elementhash[MAX_ELEMENTS];
  char name[MAX_ELEMENTS];
  char keyword[MAX_ELEMENTS];
  int Z = -1;
  int No[MAX_ELEMENTS];
  int verbose = 0;
  double value[3];

  if (mol == NULL) {
    DoeLog(0) && (eLog()<< Verbose(0) << "Molecule is not allocated in LoadMolecule(), exit.");
    performCriticalExit();
  }

  ParseForParameter(verbose,FileBuffer,"MaxTypes", 0, 1, 1, int_type, &(MaxTypes), 1, critical);
  if (MaxTypes == 0) {
    DoeLog(1) && (eLog()<< Verbose(1) << "There are no atoms according to MaxTypes in this config file." << endl);
    //performCriticalExit();
  } else {
    // prescan number of ions per type
    DoLog(0) && (Log() << Verbose(0) << "Prescanning ions per type: " << endl);
    int NoAtoms = 0;
    for (int i=0; i < MaxTypes; i++) {
      sprintf(name,"Ion_Type%i",i+1);
      ParseForParameter(verbose,FileBuffer, (const char*)name, 0, 1, 1, int_type, &No[i], 1, critical);
      ParseForParameter(verbose,FileBuffer, name, 0, 2, 1, int_type, &Z, 1, critical);
      elementhash[i] = periode->FindElement(Z);
      DoLog(1) && (Log() << Verbose(1) << i << ". Z = " << elementhash[i]->getAtomicNumber() << " with " << No[i] << " ions." << endl);
      NoAtoms += No[i];
    }
    int repetition = 0; // which repeated keyword shall be read

    // sort the lines via the LineMapping
    sprintf(name,"Ion_Type%i",MaxTypes);
    if (!ParseForParameter(verbose,FileBuffer, (const char*)name, 1, 1, 1, int_type, &value[0], 1, critical)) {
      DoeLog(0) && (eLog()<< Verbose(0) << "There are no atoms in the config file!" << endl);
      performCriticalExit();
      return;
    }
    FileBuffer->CurrentLine++;
    //Log() << Verbose(0) << FileBuffer->buffer[ FileBuffer->LineMapping[FileBuffer->CurrentLine]];
    FileBuffer->MapIonTypesInBuffer(NoAtoms);
    //for (int i=0; i<(NoAtoms < 100 ? NoAtoms : 100 < 100 ? NoAtoms : 100);++i) {
    //  Log() << Verbose(0) << FileBuffer->buffer[ FileBuffer->LineMapping[FileBuffer->CurrentLine+i]];
    //}

    map<int, atom *> AtomList[MaxTypes];
    map<int, atom *> LinearList;
    atom *neues = NULL;
    Vector position;
    if (!FastParsing) {
      // parse in trajectories
      bool status = true;
      while (status) {
        DoLog(0) && (Log() << Verbose(0) << "Currently parsing MD step " << repetition << "." << endl);
        for (int i=0; i < MaxTypes; i++) {
          sprintf(name,"Ion_Type%i",i+1);
          for(int j=0;j<No[i];j++) {
            sprintf(keyword,"%s_%i",name, j+1);
            if (repetition == 0) {
              neues = World::getInstance().createAtom();
              AtomList[i][j] = neues;
              LinearList[ FileBuffer->LineMapping[FileBuffer->CurrentLine] ] = neues;
              neues->setType(elementhash[i]); // find element type
            } else
              neues = AtomList[i][j];
            status = (status &&
                    ParseForParameter(verbose,FileBuffer, keyword, 0, 1, 1, double_type, &position[0], 1, (repetition == 0) ? critical : optional) &&
                    ParseForParameter(verbose,FileBuffer, keyword, 0, 2, 1, double_type, &position[1], 1, (repetition == 0) ? critical : optional) &&
                    ParseForParameter(verbose,FileBuffer, keyword, 0, 3, 1, double_type, &position[2], 1, (repetition == 0) ? critical : optional) &&
                    ParseForParameter(verbose,FileBuffer, keyword, 0, 4, 1, int_type, &neues->FixedIon, 1, (repetition == 0) ? critical : optional));
            if (!status)
              break;
            neues ->setPosition(position);

            // check size of vectors
            if (neues->Trajectory.R.size() <= (unsigned int)(repetition)) {
              //Log() << Verbose(0) << "Increasing size for trajectory array of " << keyword << " to " << (repetition+10) << "." << endl;
              neues->Trajectory.R.resize(repetition+10);
              neues->Trajectory.U.resize(repetition+10);
              neues->Trajectory.F.resize(repetition+10);
            }

            // put into trajectories list
            for (int d=0;d<NDIM;d++)
              neues->Trajectory.R.at(repetition)[d] = neues->at(d);

            // parse velocities if present
            if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 5, 1, double_type, &neues->AtomicVelocity[0], 1,optional))
              neues->AtomicVelocity[0] = 0.;
            if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 6, 1, double_type, &neues->AtomicVelocity[1], 1,optional))
              neues->AtomicVelocity[1] = 0.;
            if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 7, 1, double_type, &neues->AtomicVelocity[2], 1,optional))
              neues->AtomicVelocity[2] = 0.;
            for (int d=0;d<NDIM;d++)
              neues->Trajectory.U.at(repetition)[d] = neues->AtomicVelocity[d];

            // parse forces if present
            if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 8, 1, double_type, &value[0], 1,optional))
              value[0] = 0.;
            if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 9, 1, double_type, &value[1], 1,optional))
              value[1] = 0.;
            if(!ParseForParameter(verbose,FileBuffer, keyword, 1, 10, 1, double_type, &value[2], 1,optional))
              value[2] = 0.;
            for (int d=0;d<NDIM;d++)
              neues->Trajectory.F.at(repetition)[d] = value[d];

  //            Log() << Verbose(0) << "Parsed position of step " << (repetition) << ": (";
  //            for (int d=0;d<NDIM;d++)
  //              Log() << Verbose(0) << neues->Trajectory.R.at(repetition).x[d] << " ";          // next step
  //            Log() << Verbose(0) << ")\t(";
  //            for (int d=0;d<NDIM;d++)
  //              Log() << Verbose(0) << neues->Trajectory.U.at(repetition).x[d] << " ";          // next step
  //            Log() << Verbose(0) << ")\t(";
  //            for (int d=0;d<NDIM;d++)
  //              Log() << Verbose(0) << neues->Trajectory.F.at(repetition).x[d] << " ";          // next step
  //            Log() << Verbose(0) << ")" << endl;
          }
        }
        repetition++;
      }
      repetition--;
      DoLog(0) && (Log() << Verbose(0) << "Found " << repetition << " trajectory steps." << endl);
      if (repetition <= 1)  // if onyl one step, desactivate use of trajectories
        mol->MDSteps = 0;
      else
        mol->MDSteps = repetition;
    } else {
      // find the maximum number of MD steps so that we may parse last one (Ion_Type1_1 must always be present, because is the first atom)
      repetition = 0;
      while ( ParseForParameter(verbose,FileBuffer, "Ion_Type1_1", 0, 1, 1, double_type, &value[0], repetition, (repetition == 0) ? critical : optional) &&
              ParseForParameter(verbose,FileBuffer, "Ion_Type1_1", 0, 2, 1, double_type, &value[1], repetition, (repetition == 0) ? critical : optional) &&
              ParseForParameter(verbose,FileBuffer, "Ion_Type1_1", 0, 3, 1, double_type, &value[2], repetition, (repetition == 0) ? critical : optional))
        repetition++;
      DoLog(0) && (Log() << Verbose(0) << "I found " << repetition << " times the keyword Ion_Type1_1." << endl);
      // parse in molecule coordinates
      for (int i=0; i < MaxTypes; i++) {
        sprintf(name,"Ion_Type%i",i+1);
        for(int j=0;j<No[i];j++) {
          sprintf(keyword,"%s_%i",name, j+1);
          if (repetition == 0) {
            neues = World::getInstance().createAtom();
            AtomList[i][j] = neues;
            LinearList[ FileBuffer->LineMapping[FileBuffer->CurrentLine] ] = neues;
            neues->setType(elementhash[i]); // find element type
          } else
            neues = AtomList[i][j];
          // then parse for each atom the coordinates as often as present
          ParseForParameter(verbose,FileBuffer, keyword, 0, 1, 1, double_type, &position[0], repetition,critical);
          ParseForParameter(verbose,FileBuffer, keyword, 0, 2, 1, double_type, &position[1], repetition,critical);
          ParseForParameter(verbose,FileBuffer, keyword, 0, 3, 1, double_type, &position[2], repetition,critical);
          neues->setPosition(position);
          ParseForParameter(verbose,FileBuffer, keyword, 0, 4, 1, int_type, &neues->FixedIon, repetition,critical);
          if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 5, 1, double_type, &neues->AtomicVelocity[0], repetition,optional))
            neues->AtomicVelocity[0] = 0.;
          if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 6, 1, double_type, &neues->AtomicVelocity[1], repetition,optional))
            neues->AtomicVelocity[1] = 0.;
          if(!ParseForParameter(verbose,FileBuffer, keyword, 0, 7, 1, double_type, &neues->AtomicVelocity[2], repetition,optional))
            neues->AtomicVelocity[2] = 0.;
          // here we don't care if forces are present (last in trajectories is always equal to current position)
          neues->setType(elementhash[i]); // find element type
          mol->AddAtom(neues);
        }
      }
    }
    // put atoms into the molecule in their original order
    for(map<int, atom*>::iterator runner = LinearList.begin(); runner != LinearList.end(); ++runner) {
      mol->AddAtom(runner->second);
    }
  }
};

/** Stores all elements of config structure from which they can be re-read.
 * \param *filename name of file
 * \param *periode pointer to a periodentafel class with all elements
 * \param *mol pointer to molecule containing all atoms of the molecule
 */
bool config::Save(const char * const filename, const periodentafel * const periode, molecule * const mol) const
{
  bool result = true;
  const Matrix &domain = World::getInstance().getDomain().getM();
  ofstream * const output = new ofstream(filename, ios::out);
  if (output != NULL) {
    *output << "# ParallelCarParinello - main configuration file - created with molecuilder" << endl;
    *output << endl;
    *output << "mainname\t" << config::mainname << "\t# programm name (for runtime files)" << endl;
    *output << "defaultpath\t" << config::defaultpath << "\t# where to put files during runtime" << endl;
    *output << "pseudopotpath\t" << config::pseudopotpath << "\t# where to find pseudopotentials" << endl;
    *output << endl;
    *output << "ProcPEGamma\t" << config::ProcPEGamma << "\t# for parallel computing: share constants" << endl;
    *output << "ProcPEPsi\t" << config::ProcPEPsi << "\t# for parallel computing: share wave functions" << endl;
    *output << "DoOutVis\t" << config::DoOutVis << "\t# Output data for OpenDX" << endl;
    *output << "DoOutMes\t" << config::DoOutMes << "\t# Output data for measurements" << endl;
    *output << "DoOutOrbitals\t" << config::DoOutOrbitals << "\t# Output all Orbitals" << endl;
    *output << "DoOutCurr\t" << config::DoOutCurrent << "\t# Ouput current density for OpenDx" << endl;
    *output << "DoOutNICS\t" << config::DoOutNICS << "\t# Output Nucleus independent current shieldings" << endl;
    *output << "DoPerturbation\t" << config::DoPerturbation << "\t# Do perturbation calculate and determine susceptibility and shielding" << endl;
    *output << "DoFullCurrent\t" << config::DoFullCurrent << "\t# Do full perturbation" << endl;
    *output << "DoConstrainedMD\t" << config::DoConstrainedMD << "\t# Do perform a constrained (>0, relating to current MD step) instead of unconstrained (0) MD" << endl;
    *output << "Thermostat\t" << Thermostats->activeThermostat->name() << "\t";
    *output << Thermostats->activeThermostat->writeParams();
    *output << "\t# Which Thermostat and its parameters to use in MD case." << endl;
    *output << "CommonWannier\t" << config::CommonWannier << "\t# Put virtual centers at indivual orbits, all common, merged by variance, to grid point, to cell center" << endl;
    *output << "SawtoothStart\t" << config::SawtoothStart << "\t# Absolute value for smooth transition at cell border " << endl;
    *output << "VectorPlane\t" << config::VectorPlane << "\t# Cut plane axis (x, y or z: 0,1,2) for two-dim current vector plot" << endl;
    *output << "VectorCut\t" << config::VectorCut << "\t# Cut plane axis value" << endl;
    *output << "AddGramSch\t" << config::UseAddGramSch << "\t# Additional GramSchmidtOrtogonalization to be safe" << endl;
    *output << "Seed\t\t" << config::Seed << "\t# initial value for random seed for Psi coefficients" << endl;
    *output << endl;
    *output << "MaxOuterStep\t" << config::MaxOuterStep << "\t# number of MolecularDynamics/Structure optimization steps" << endl;
    *output << "Deltat\t" << config::Deltat << "\t# time per MD step" << endl;
    *output << "OutVisStep\t" << config::OutVisStep << "\t# Output visual data every ...th step" << endl;
    *output << "OutSrcStep\t" << config::OutSrcStep << "\t# Output \"restart\" data every ..th step" << endl;
    *output << "TargetTemp\t" << Thermostats->TargetTemp << "\t# Target temperature" << endl;
    *output << "MaxPsiStep\t" << config::MaxPsiStep << "\t# number of Minimisation steps per state (0 - default)" << endl;
    *output << "EpsWannier\t" << config::EpsWannier << "\t# tolerance value for spread minimisation of orbitals" << endl;
    *output << endl;
    *output << "# Values specifying when to stop" << endl;
    *output << "MaxMinStep\t" << config::MaxMinStep << "\t# Maximum number of steps" << endl;
    *output << "RelEpsTotalE\t" << config::RelEpsTotalEnergy << "\t# relative change in total energy" << endl;
    *output << "RelEpsKineticE\t" << config::RelEpsKineticEnergy << "\t# relative change in kinetic energy" << endl;
    *output << "MaxMinStopStep\t" << config::MaxMinStopStep << "\t# check every ..th steps" << endl;
    *output << "MaxMinGapStopStep\t" << config::MaxMinGapStopStep << "\t# check every ..th steps" << endl;
    *output << endl;
    *output << "# Values specifying when to stop for INIT, otherwise same as above" << endl;
    *output << "MaxInitMinStep\t" << config::MaxInitMinStep << "\t# Maximum number of steps" << endl;
    *output << "InitRelEpsTotalE\t" << config::InitRelEpsTotalEnergy << "\t# relative change in total energy" << endl;
    *output << "InitRelEpsKineticE\t" << config::InitRelEpsKineticEnergy << "\t# relative change in kinetic energy" << endl;
    *output << "InitMaxMinStopStep\t" << config::InitMaxMinStopStep << "\t# check every ..th steps" << endl;
    *output << "InitMaxMinGapStopStep\t" << config::InitMaxMinGapStopStep << "\t# check every ..th steps" << endl;
    *output << endl;
    *output << "BoxLength\t\t\t# (Length of a unit cell)" << endl;
    *output << domain.at(0,0) << "\t" << endl;
    *output << domain.at(1,0) << "\t" << domain.at(1,1) << "\t" << endl;
    *output << domain.at(2,0) << "\t" << domain.at(2,1) << "\t" << domain.at(2,2) << "\t" << endl;
    // FIXME
    *output << endl;
    *output << "ECut\t\t" << config::ECut << "\t# energy cutoff for discretization in Hartrees" << endl;
    *output << "MaxLevel\t" << config::MaxLevel << "\t# number of different levels in the code, >=2" << endl;
    *output << "Level0Factor\t" << config::Lev0Factor << "\t# factor by which node number increases from S to 0 level" << endl;
    *output << "RiemannTensor\t" << config::RiemannTensor << "\t# (Use metric)" << endl;
    switch (config::RiemannTensor) {
      case 0: //UseNoRT
        break;
      case 1: // UseRT
        *output << "RiemannLevel\t" << config::RiemannLevel << "\t# Number of Riemann Levels" << endl;
        *output << "LevRFactor\t" << config::LevRFactor << "\t# factor by which node number increases from 0 to R level from" << endl;
        break;
    }
    *output << "PsiType\t\t" << config::PsiType << "\t# 0 - doubly occupied, 1 - SpinUp,SpinDown" << endl;
    // write out both types for easier changing afterwards
  //  switch (PsiType) {
  //    case 0:
        *output << "MaxPsiDouble\t" << config::MaxPsiDouble << "\t# here: specifying both maximum number of SpinUp- and -Down-states" << endl;
  //      break;
  //    case 1:
        *output << "PsiMaxNoUp\t" << config::PsiMaxNoUp << "\t# here: specifying maximum number of SpinUp-states" << endl;
        *output << "PsiMaxNoDown\t" << config::PsiMaxNoDown << "\t# here: specifying maximum number of SpinDown-states" << endl;
  //      break;
  //  }
    *output << "AddPsis\t\t" << config::AddPsis << "\t# Additional unoccupied Psis for bandgap determination" << endl;
    *output << endl;
    *output << "RCut\t\t" << config::RCut << "\t# R-cut for the ewald summation" << endl;
    *output << "StructOpt\t" << config::StructOpt << "\t# Do structure optimization beforehand" << endl;
    *output << "IsAngstroem\t" << config::IsAngstroem << "\t# 0 - Bohr, 1 - Angstroem" << endl;
    *output << "RelativeCoord\t" << config::RelativeCoord << "\t# whether ion coordinates are relative (1) or absolute (0)" << endl;
    *output << "MaxTypes\t" << mol->getElementCount() <<  "\t# maximum number of different ion types" << endl;
    *output << endl;
    result = result && mol->Checkout(output);
    if (mol->MDSteps <=1 )
      result = result && mol->Output(output);
    else
      result = result && mol->OutputTrajectories(output);
    output->close();
    output->clear();
    delete(output);
    return result;
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open output file:" << filename << endl);
    return false;
  }
};

/** Stores all elements in a MPQC input file.
 * Note that this format cannot be parsed again.
 * \param *filename name of file (without ".in" suffix!)
 * \param *mol pointer to molecule containing all atoms of the molecule
 */
bool config::SaveMPQC(const char * const filename, const molecule * const mol) const
{
  Vector *center = NULL;
  ofstream *output = NULL;

  // first without hessian
  {
    stringstream * const fname = new stringstream;;
    *fname << filename << ".in";
    output = new ofstream(fname->str().c_str(), ios::out);
    if (output == NULL) {
      DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open mpqc output file:" << fname << endl);
      delete(fname);
      return false;
    }
    *output << "% Created by MoleCuilder" << endl;
    *output << "mpqc: (" << endl;
    *output << "\tsavestate = no" << endl;
    *output << "\tdo_gradient = yes" << endl;
    *output << "\tmole<MBPT2>: (" << endl;
    *output << "\t\tmaxiter = 200" << endl;
    *output << "\t\tbasis = $:basis" << endl;
    *output << "\t\tmolecule = $:molecule" << endl;
    *output << "\t\treference<CLHF>: (" << endl;
    *output << "\t\t\tbasis = $:basis" << endl;
    *output << "\t\t\tmolecule = $:molecule" << endl;
    *output << "\t\t)" << endl;
    *output << "\t)" << endl;
    *output << ")" << endl;
    *output << "molecule<Molecule>: (" << endl;
    *output << "\tunit = " << (IsAngstroem ? "angstrom" : "bohr" ) << endl;
    *output << "\t{ atoms geometry } = {" << endl;
    center = mol->DetermineCenterOfAll();
    // output of atoms
    mol->ActOnAllAtoms( &atom::OutputMPQCLine, (ostream * const) output, (const Vector *)center);
    delete(center);
    *output << "\t}" << endl;
    *output << ")" << endl;
    *output << "basis<GaussianBasisSet>: (" << endl;
    *output << "\tname = \"" << basis << "\"" << endl;
    *output << "\tmolecule = $:molecule" << endl;
    *output << ")" << endl;
    output->close();
    delete(output);
    delete(fname);
  }

  // second with hessian
  {
    stringstream * const fname = new stringstream;
    *fname << filename << ".hess.in";
    output = new ofstream(fname->str().c_str(), ios::out);
    if (output == NULL) {
      DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open mpqc hessian output file:" << fname << endl);
      delete(fname);
      return false;
    }
    *output << "% Created by MoleCuilder" << endl;
    *output << "mpqc: (" << endl;
    *output << "\tsavestate = no" << endl;
    *output << "\tdo_gradient = yes" << endl;
    *output << "\tmole<CLHF>: (" << endl;
    *output << "\t\tmaxiter = 200" << endl;
    *output << "\t\tbasis = $:basis" << endl;
    *output << "\t\tmolecule = $:molecule" << endl;
    *output << "\t)" << endl;
    *output << "\tfreq<MolecularFrequencies>: (" << endl;
    *output << "\t\tmolecule=$:molecule" << endl;
    *output << "\t)" << endl;
    *output << ")" << endl;
    *output << "molecule<Molecule>: (" << endl;
    *output << "\tunit = " << (IsAngstroem ? "angstrom" : "bohr" ) << endl;
    *output << "\t{ atoms geometry } = {" << endl;
    center = mol->DetermineCenterOfAll();
    // output of atoms
    mol->ActOnAllAtoms( &atom::OutputMPQCLine, (ostream * const) output, (const Vector *)center);
    delete(center);
    *output << "\t}" << endl;
    *output << ")" << endl;
    *output << "basis<GaussianBasisSet>: (" << endl;
    *output << "\tname = \"3-21G\"" << endl;
    *output << "\tmolecule = $:molecule" << endl;
    *output << ")" << endl;
    output->close();
    delete(output);
    delete(fname);
  }

  return true;
};

/** Stores all atoms from all molecules in a PDB input file.
 * Note that this format cannot be parsed again.
 * \param *filename name of file (without ".in" suffix!)
 * \param *MolList pointer to MoleculeListClass containing all atoms
 */
bool config::SavePDB(const char * const filename, const MoleculeListClass * const MolList) const
{
  int AtomNo = -1;
  int MolNo = 0;
  FILE *f = NULL;

  char name[MAXSTRINGSIZE];
  strncpy(name, filename, MAXSTRINGSIZE-1);
  strncat(name, ".pdb", MAXSTRINGSIZE-(strlen(name)+1));
  f = fopen(name, "w" );
  if (f == NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open pdb output file:" << name << endl);
    return false;
  }
  fprintf(f, "# Created by MoleCuilder\n");

  for (MoleculeList::const_iterator MolRunner = MolList->ListOfMolecules.begin(); MolRunner != MolList->ListOfMolecules.end(); MolRunner++) {
    int *elementNo = new int[MAX_ELEMENTS];
    for (int i=0;i<MAX_ELEMENTS;i++)
      elementNo[i] = 0;
    AtomNo = 0;
    for (molecule::const_iterator iter = (*MolRunner)->begin(); iter != (*MolRunner)->end(); ++iter) {
      sprintf(name, "%2s%2d",(*iter)->getType()->getSymbol().c_str(), elementNo[(*iter)->getType()->getAtomicNumber()]);
      elementNo[(*iter)->getType()->getAtomicNumber()] = (elementNo[(*iter)->getType()->getAtomicNumber()]+1) % 100;   // confine to two digits
      fprintf(f,
             "ATOM %6u %-4s %4s%c%4u    %8.3f%8.3f%8.3f%6.2f%6.2f      %4s%2s%2s\n",
             (*iter)->nr,                /* atom serial number */
             name,         /* atom name */
             (*MolRunner)->name,      /* residue name */
             'a'+(unsigned char)(AtomNo % 26),           /* letter for chain */
             MolNo,         /* residue sequence number */
             (*iter)->at(0),                 /* position X in Angstroem */
             (*iter)->at(1),                 /* position Y in Angstroem */
             (*iter)->at(2),                 /* position Z in Angstroem */
             (double)(*iter)->getType()->getValence(),         /* occupancy */
             (double)(*iter)->getType()->getNoValenceOrbitals(),          /* temperature factor */
             "0",            /* segment identifier */
             (*iter)->getType()->getSymbol().c_str(),    /* element symbol */
             "0");           /* charge */
      AtomNo++;
    }
    delete[](elementNo);
    MolNo++;
  }
  fclose(f);

  return true;
};

/** Stores all atoms in a PDB input file.
 * Note that this format cannot be parsed again.
 * \param *filename name of file (without ".in" suffix!)
 * \param *mol pointer to molecule
 */
bool config::SavePDB(const char * const filename, const molecule * const mol) const
{
  int AtomNo = -1;
  FILE *f = NULL;

  int *elementNo = new int[MAX_ELEMENTS];
  for (int i=0;i<MAX_ELEMENTS;i++)
    elementNo[i] = 0;
  char name[MAXSTRINGSIZE];
  strncpy(name, filename, MAXSTRINGSIZE-1);
  strncat(name, ".pdb", MAXSTRINGSIZE-(strlen(name)+1));
  f = fopen(name, "w" );
  if (f == NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open pdb output file:" << name << endl);
    delete[](elementNo);
    return false;
  }
  fprintf(f, "# Created by MoleCuilder\n");

  AtomNo = 0;
  for (molecule::const_iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    sprintf(name, "%2s%2d",(*iter)->getType()->getSymbol().c_str(), elementNo[(*iter)->getType()->getAtomicNumber()]);
    elementNo[(*iter)->getType()->getAtomicNumber()] = (elementNo[(*iter)->getType()->getAtomicNumber()]+1) % 100;   // confine to two digits
    fprintf(f,
           "ATOM %6u %-4s %4s%c%4u    %8.3f%8.3f%8.3f%6.2f%6.2f      %4s%2s%2s\n",
           (*iter)->nr,                /* atom serial number */
           name,         /* atom name */
           mol->name,      /* residue name */
           'a'+(unsigned char)(AtomNo % 26),           /* letter for chain */
           0,         /* residue sequence number */
           (*iter)->at(0),                 /* position X in Angstroem */
           (*iter)->at(1),                 /* position Y in Angstroem */
           (*iter)->at(2),                 /* position Z in Angstroem */
           (double)(*iter)->getType()->getValence(),         /* occupancy */
           (double)(*iter)->getType()->getNoValenceOrbitals(),          /* temperature factor */
           "0",            /* segment identifier */
           (*iter)->getType()->getSymbol().c_str(),    /* element symbol */
           "0");           /* charge */
    AtomNo++;
  }
  fclose(f);
  delete[](elementNo);

  return true;
};

/** Stores all atoms in a TREMOLO data input file.
 * Note that this format cannot be parsed again.
 * Note that TREMOLO does not like Id starting at 0, but at 1. Atoms with Id 0 are discarded!
 * \param *filename name of file (without ".in" suffix!)
 * \param *mol pointer to molecule
 */
bool config::SaveTREMOLO(const char * const filename, const molecule * const mol) const
{
  ofstream *output = NULL;
  stringstream * const fname = new stringstream;

  *fname << filename << ".data";
  output = new ofstream(fname->str().c_str(), ios::out);
  if (output == NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open tremolo output file:" << fname << endl);
    delete(fname);
    return false;
  }

  // scan maximum number of neighbours
  int MaxNeighbours = 0;
  for (molecule::const_iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    const int count = (*iter)->ListOfBonds.size();
    if (MaxNeighbours < count)
      MaxNeighbours = count;
  }
  *output << "# ATOMDATA Id name resName resSeq x=3 Charge type neighbors=" << MaxNeighbours << endl;

  for (molecule::const_iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    *output << (*iter)->nr << "\t";
    *output << (*iter)->getName() << "\t";
    *output << mol->name << "\t";
    *output << 0 << "\t";
    *output << (*iter)->at(0) << "\t" << (*iter)->at(1) << "\t" << (*iter)->at(2) << "\t";
    *output << static_cast<double>((*iter)->getType()->getValence()) << "\t";
    *output << (*iter)->getType()->getSymbol() << "\t";
    for (BondList::iterator runner = (*iter)->ListOfBonds.begin(); runner != (*iter)->ListOfBonds.end(); runner++)
      *output << (*runner)->GetOtherAtom(*iter)->nr << "\t";
    for(int i=(*iter)->ListOfBonds.size(); i < MaxNeighbours; i++)
      *output << "-\t";
    *output << endl;
  }
  output->flush();
  output->close();
  delete(output);
  delete(fname);

  return true;
};

/** Stores all atoms from all molecules in a TREMOLO data input file.
 * Note that this format cannot be parsed again.
 * Note that TREMOLO does not like Id starting at 0, but at 1. Atoms with Id 0 are discarded!
 * \param *filename name of file (without ".in" suffix!)
 * \param *MolList pointer to MoleculeListClass containing all atoms
 */
bool config::SaveTREMOLO(const char * const filename, const MoleculeListClass * const MolList) const
{
  Info FunctionInfo(__func__);
  ofstream *output = NULL;
  stringstream * const fname = new stringstream;

  *fname << filename << ".data";
  output = new ofstream(fname->str().c_str(), ios::out);
  if (output == NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open tremolo output file:" << fname << endl);
    delete(fname);
    return false;
  }

  // scan maximum number of neighbours
  int MaxNeighbours = 0;
  for (MoleculeList::const_iterator MolWalker = MolList->ListOfMolecules.begin(); MolWalker != MolList->ListOfMolecules.end(); MolWalker++) {
    for (molecule::const_iterator iter = (*MolWalker)->begin(); iter != (*MolWalker)->end(); ++iter) {
      const int count = (*iter)->ListOfBonds.size();
      if (MaxNeighbours < count)
        MaxNeighbours = count;
    }
  }
  *output << "# ATOMDATA Id name resName resSeq x=3 Charge type neighbors=" << MaxNeighbours << endl;

  // create global to local id map
  map<int, int> LocalNotoGlobalNoMap;
  {
    unsigned int MolCounter = 0;
    int AtomNo = 1;
    for (MoleculeList::const_iterator MolWalker = MolList->ListOfMolecules.begin(); MolWalker != MolList->ListOfMolecules.end(); MolWalker++) {
      for(molecule::iterator AtomRunner = (*MolWalker)->begin(); AtomRunner != (*MolWalker)->end(); ++AtomRunner) {
        LocalNotoGlobalNoMap.insert( pair<int,int>((*AtomRunner)->getId(), AtomNo++) );
      }
      MolCounter++;
    }
    ASSERT(MolCounter == MolList->ListOfMolecules.size(), "SaveTREMOLO: LocalNotoGlobalNoMap[] has not been correctly initialized for each molecule");
  }

  // write the file
  {
    int MolCounter = 0;
    int AtomNo = 0;
    for (MoleculeList::const_iterator MolWalker = MolList->ListOfMolecules.begin(); MolWalker != MolList->ListOfMolecules.end(); MolWalker++) {
      for (molecule::const_iterator iter = (*MolWalker)->begin(); iter != (*MolWalker)->end(); ++iter) {
        *output << LocalNotoGlobalNoMap[ (*iter)->getId() ] << "\t";
        *output << (*iter)->getName() << "\t";
        *output << (*MolWalker)->name << "\t";
        *output << MolCounter+1 << "\t";
        *output << (*iter)->at(0) << "\t" << (*iter)->at(1) << "\t" << (*iter)->at(2) << "\t";
        *output << (double)(*iter)->getType()->getValence() << "\t";
        *output << (*iter)->getType()->getSymbol() << "\t";
        for (BondList::iterator runner = (*iter)->ListOfBonds.begin(); runner != (*iter)->ListOfBonds.end(); runner++)
          *output << LocalNotoGlobalNoMap[ (*runner)->GetOtherAtom((*iter))->getId() ] << "\t";
        for(int i=(*iter)->ListOfBonds.size(); i < MaxNeighbours; i++)
          *output << "-\t";
        *output << endl;
        AtomNo++;
      }
      MolCounter++;
    }
  }

  // store & free
  output->flush();
  output->close();
  delete(output);
  delete(fname);

  return true;
};


/** Tries given filename or standard on saving the config file.
 * \param *ConfigFileName name of file
 * \param *periode pointer to periodentafel structure with all the elements
 * \param *molecules list of molecules structure with all the atoms and coordinates
 */
void config::SaveAll(char *ConfigFileName, periodentafel *periode, MoleculeListClass *molecules)
{
  char filename[MAXSTRINGSIZE];
  ofstream output;
  molecule *mol = NULL;

  // first save as PDB data
  if (ConfigFileName != NULL)
    strcpy(filename, ConfigFileName);
  if (output == NULL)
    strcpy(filename,"main_pcp_linux");
  Log() << Verbose(0) << "Saving as pdb input ... " << endl;
  if (SavePDB(filename, molecules))
    Log() << Verbose(0) << "\t... done." << endl;
  else
    Log() << Verbose(0) << "\t... failed." << endl;

  // then save as tremolo data file
  if (ConfigFileName != NULL)
    strcpy(filename, ConfigFileName);
  if (output == NULL)
    strcpy(filename,"main_pcp_linux");
  Log() << Verbose(0) << "Saving as tremolo data input ... " << endl;
  if (SaveTREMOLO(filename, molecules))
    Log() << Verbose(0) << "\t... done." << endl;
  else
    Log() << Verbose(0) << "\t... failed." << endl;

  // translate each to its center and merge all molecules in MoleculeListClass into this molecule
  int N = molecules->ListOfMolecules.size();
  if (N != 1) { // don't do anything in case of only one molecule (shifts mol ids otherwise)
    int *src = new int[N];
    N=0;
    for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++) {
      src[N++] = (*ListRunner)->IndexNr;
    }
    mol = World::getInstance().createMolecule();
    mol->SetNameFromFilename(ConfigFileName);
    //mol->CalculateOrbitals(*this);
    delete[](src);
  } else {
    if (!molecules->ListOfMolecules.empty()) {
      mol = *(molecules->ListOfMolecules.begin());
      mol->doCountAtoms();
      //mol->CalculateOrbitals(*this);
    } else {
      DoeLog(1) && (eLog() << Verbose(1) << "There are no molecules to save!" << endl);
    }
  }

  Log() << Verbose(0) << "Storing configuration ... " << endl;
  // get correct valence orbitals
  if (ConfigFileName != NULL) { // test the file name
    strcpy(filename, ConfigFileName);
    output.open(filename, ios::trunc);
  } else if (strlen(configname) != 0) {
    strcpy(filename, configname);
    output.open(configname, ios::trunc);
    } else {
      strcpy(filename, DEFAULTCONFIG);
      output.open(DEFAULTCONFIG, ios::trunc);
    }
  output.close();
  output.clear();
  Log() << Verbose(0) << "Saving of config file ... " << endl;
  if (Save(filename, periode, mol))
    Log() << Verbose(0) << "\t... successful." << endl;
  else
    Log() << Verbose(0) << "\t... failed." << endl;

  // and save to xyz file
  if (ConfigFileName != NULL) {
    strcpy(filename, ConfigFileName);
    strcat(filename, ".xyz");
    output.open(filename, ios::trunc);
  }
  if (output == NULL) {
    strcpy(filename,"main_pcp_linux");
    strcat(filename, ".xyz");
    output.open(filename, ios::trunc);
  }
  Log() << Verbose(0) << "Saving of XYZ file ... " << endl;
  if (mol->MDSteps <= 1) {
    if (mol->OutputXYZ(&output))
      Log() << Verbose(0) << "\t... successful." << endl;
    else
      Log() << Verbose(0) << "\t... failed." << endl;
  } else {
    if (mol->OutputTrajectoriesXYZ(&output))
      Log() << Verbose(0) << "\t... successful." << endl;
    else
      Log() << Verbose(0) << "\t... failed." << endl;
  }
  output.close();
  output.clear();

  // and save as MPQC configuration
  if (ConfigFileName != NULL)
    strcpy(filename, ConfigFileName);
  if (output == NULL)
    strcpy(filename,"main_pcp_linux");
  Log() << Verbose(0) << "Saving as mpqc input .. " << endl;
  if (SaveMPQC(filename, mol))
    Log() << Verbose(0) << "\t... done." << endl;
  else
    Log() << Verbose(0) << "\t... failed." << endl;

  // don't destroy molecule as it contains all our atoms
  //World::getInstance().destroyMolecule(mol);
};

/** Reads parameter from a parsed file.
 * The file is either parsed for a certain keyword or if null is given for
 * the value in row yth and column xth. If the keyword was necessity#critical,
 * then an error is thrown and the programme aborted.
 * \warning value is modified (both in contents and position)!
 * \param verbose 1 - print found value to stderr, 0 - don't
 * \param *file file to be parsed
 * \param name Name of value in file (at least 3 chars!)
 * \param sequential 1 - do not reset file pointer to begin of file, 0 - set to beginning
 *        (if file is sequentially parsed this can be way faster! However, beware of multiple values per line, as whole line is read -
 *         best approach: 0 0 0 1 (not resetted only on last value of line) - and of yth, which is now
 *         counted from this unresetted position!)
 * \param xth Which among a number of parameters it is (in grid case it's row number as grid is read as a whole!)
 * \param yth In grid case specifying column number, otherwise the yth \a name matching line
 * \param type Type of the Parameter to be read
 * \param value address of the value to be read (must have been allocated)
 * \param repetition determines, if the keyword appears multiply in the config file, which repetition shall be parsed, i.e. 1 if not multiply
 * \param critical necessity of this keyword being specified (optional, critical)
 * \return 1 - found, 0 - not found
 * \note Routine is taken from the pcp project and hack-a-slack adapted to C++
 */
int ParseForParameter(const int verbose, ifstream * const file, const char * const name, const int sequential, const int xth, const int yth, const int type, void * value, const int repetition, const int critical) {
  int i = 0;
  int j = 0;  // loop variables
  int length = 0;
  int maxlength = -1;
  long file_position = file->tellg(); // mark current position
  char *dummy1 = NULL;
  char *dummy = NULL;
  char free_dummy[MAXSTRINGSIZE];    // pointers in the line that is read in per step
  dummy1 = free_dummy;

  //fprintf(stderr,"Parsing for %s\n",name);
  if (repetition == 0)
    //Error(SomeError, "ParseForParameter(): argument repetition must not be 0!");
    return 0;

  int line = 0; // marks line where parameter was found
  int found = (type >= grid) ? 0 : (-yth + 1);  // marks if yth parameter name was found
  while((found != repetition)) {
    dummy1 = dummy = free_dummy;
    do {
      file->getline(dummy1, 256); // Read the whole line
      if (file->eof()) {
        if ((critical) && (found == 0)) {
          //Error(InitReading, name);
          fprintf(stderr,"Error:InitReading, critical %s not found\n", name);
          exit(255);
        } else {
          //if (!sequential)
          file->clear();
          file->seekg(file_position, ios::beg);  // rewind to start position
          return 0;
        }
      }
      line++;
    } while (dummy != NULL && dummy1 != NULL && ((dummy1[0] == '#') || (dummy1[0] == '\0'))); // skip commentary and empty lines

    // C++ getline removes newline at end, thus re-add
    if ((dummy1 != NULL) && (strchr(dummy1,'\n') == NULL)) {
      i = strlen(dummy1);
      dummy1[i] = '\n';
      dummy1[i+1] = '\0';
    }
    //fprintf(stderr,"line %i ends at %i, newline at %i\n", line, strlen(dummy1), strchr(dummy1,'\n')-free_dummy);

    if (dummy1 == NULL) {
      if (verbose) fprintf(stderr,"Error reading line %i\n",line);
    } else {
      //fprintf(stderr,"Now parsing the line %i: %s\n", line, dummy1);
    }
    // Seek for possible end of keyword on line if given ...
    if (name != NULL) {
      dummy = strchr(dummy1,'\t');  // set dummy on first tab or space which ever's nearer
      if (dummy == NULL) {
        dummy = strchr(dummy1, ' ');  // if not found seek for space
        while ((dummy != NULL) && ((*dummy == '\t') || (*dummy == ' ')))    // skip some more tabs and spaces if necessary
          dummy++;
      }
      if (dummy == NULL) {
        dummy = strchr(dummy1, '\n'); // set on line end then (whole line = keyword)
        //fprintf(stderr,"Error: Cannot find tabs or spaces on line %i in search for %s\n", line, name);
        //Error(FileOpenParams, NULL);
      } else {
        //fprintf(stderr,"found tab at %i\n",(char *)dummy-(char *)dummy1);
      }
    } else dummy = dummy1;
    // ... and check if it is the keyword!
    //fprintf(stderr,"name %p, dummy %i/%c, dummy1 %i/%c, strlen(name) %i\n", &name, dummy, *dummy, dummy1, *dummy1, strlen(name));
    if ((name == NULL) || (((dummy-dummy1 >= 3) && (strncmp(dummy1, name, strlen(name)) == 0)) && ((unsigned int)(dummy-dummy1) == strlen(name)))) {
      found++; // found the parameter!
      //fprintf(stderr,"found %s at line %i between %i and %i\n", name, line, dummy1, dummy);

      if (found == repetition) {
        for (i=0;i<xth;i++) { // i = rows
          if (type >= grid) {
            // grid structure means that grid starts on the next line, not right after keyword
            dummy1 = dummy = free_dummy;
            do {
              file->getline(dummy1, 256); // Read the whole line, skip commentary and empty ones
              if (file->eof()) {
                if ((critical) && (found == 0)) {
                  //Error(InitReading, name);
                  fprintf(stderr,"Error:InitReading, critical %s not found\n", name);
                  exit(255);
                } else {
                  //if (!sequential)
                  file->clear();
                  file->seekg(file_position, ios::beg);  // rewind to start position
                  return 0;
                }
              }
              line++;
            } while ((dummy1[0] == '#') || (dummy1[0] == '\n'));
            if (dummy1 == NULL){
              if (verbose) fprintf(stderr,"Error reading line %i\n", line);
            } else {
              //fprintf(stderr,"Reading next line %i: %s\n", line, dummy1);
            }
          } else { // simple int, strings or doubles start in the same line
            while ((*dummy == '\t') || (*dummy == ' '))   // skip interjacent tabs and spaces
              dummy++;
          }
          // C++ getline removes newline at end, thus re-add
          if ((dummy1 != NULL) && (strchr(dummy1,'\n') == NULL)) {
            j = strlen(dummy1);
            dummy1[j] = '\n';
            dummy1[j+1] = '\0';
          }

          int start = (type >= grid) ? 0 : yth-1 ;
          for (j=start;j<yth;j++) { // j = columns
            // check for lower triangular area and upper triangular area
            if ( ((i > j) && (type == upper_trigrid)) || ((j > i) && (type == lower_trigrid))) {
              *((double *)value) = 0.0;
              fprintf(stderr,"%f\t",*((double *)value));
              value = (void *)((long)value + sizeof(double));
              //value += sizeof(double);
            } else {
              // otherwise we must skip all interjacent tabs and spaces and find next value
              dummy1 = dummy;
              dummy = strchr(dummy1, '\t'); // seek for tab or space
              if (dummy == NULL)
                dummy = strchr(dummy1, ' ');  // if not found seek for space
              if (dummy == NULL) { // if still zero returned ...
                dummy = strchr(dummy1, '\n');  // ... at line end then
                if ((j < yth-1) && (type < 4)) {  // check if xth value or not yet
                  if (critical) {
                    if (verbose) fprintf(stderr,"Error: EoL at %i and still missing %i value(s) for parameter %s\n", line, yth-j, name);
                    //return 0;
                    exit(255);
                    //Error(FileOpenParams, NULL);
                  } else {
                    //if (!sequential)
                    file->clear();
                    file->seekg(file_position, ios::beg);  // rewind to start position
                    return 0;
                  }
                }
              } else {
                //fprintf(stderr,"found tab at %i\n",(char *)dummy-(char *)free_dummy);
              }
              if (*dummy1 == '#') {
                // found comment, skipping rest of line
                //if (verbose) fprintf(stderr,"Error: '#' at %i and still missing %i value(s) for parameter %s\n", line, yth-j, name);
                if (!sequential) { // here we need it!
                  file->seekg(file_position, ios::beg);  // rewind to start position
                }
                return 0;
              }
              //fprintf(stderr,"value from %i to %i\n",(char *)dummy1-(char *)free_dummy,(char *)dummy-(char *)free_dummy);
              switch(type) {
                case (row_int):
                  *((int *)value) = atoi(dummy1);
                  if ((verbose) && (i==0) && (j==0)) fprintf(stderr,"%s = ", name);
                  if (verbose) fprintf(stderr,"%i\t",*((int *)value));
                    value = (void *)((long)value + sizeof(int));
                    //value += sizeof(int);
                  break;
                case(row_double):
                case(grid):
                case(lower_trigrid):
                case(upper_trigrid):
                  *((double *)value) = atof(dummy1);
                  if ((verbose) && (i==0) && (j==0)) fprintf(stderr,"%s = ", name);
                  if (verbose) fprintf(stderr,"%lg\t",*((double *)value));
                  value = (void *)((long)value + sizeof(double));
                  //value += sizeof(double);
                  break;
                case(double_type):
                  *((double *)value) = atof(dummy1);
                  if ((verbose) && (i == xth-1)) fprintf(stderr,"%s = %lg\n", name, *((double *) value));
                  //value += sizeof(double);
                  break;
                case(int_type):
                  *((int *)value) = atoi(dummy1);
                  if ((verbose) && (i == xth-1)) fprintf(stderr,"%s = %i\n", name, *((int *) value));
                  //value += sizeof(int);
                  break;
                default:
                case(string_type):
                  if (value != NULL) {
                    //if (maxlength == -1) maxlength = strlen((char *)value); // get maximum size of string array
                    maxlength = MAXSTRINGSIZE;
                    length = maxlength > (dummy-dummy1) ? (dummy-dummy1) : maxlength; // cap at maximum
                    strncpy((char *)value, dummy1, length);  // copy as much
                    ((char *)value)[length] = '\0';  // and set end marker
                    if ((verbose) && (i == xth-1)) fprintf(stderr,"%s is '%s' (%i chars)\n",name,((char *) value), length);
                    //value += sizeof(char);
                  } else {
                  }
                break;
              }
            }
            while (*dummy == '\t')
              dummy++;
          }
        }
      }
    }
  }
  if ((type >= row_int) && (verbose))
    fprintf(stderr,"\n");
  if (!sequential) {
    file->clear();
    file->seekg(file_position, ios::beg);  // rewind to start position
  }
  //fprintf(stderr, "End of Parsing\n\n");

  return (found); // true if found, false if not
}


/** Reads parameter from a parsed file.
 * The file is either parsed for a certain keyword or if null is given for
 * the value in row yth and column xth. If the keyword was necessity#critical,
 * then an error is thrown and the programme aborted.
 * \warning value is modified (both in contents and position)!
 * \param verbose 1 - print found value to stderr, 0 - don't
 * \param *FileBuffer pointer to buffer structure
 * \param name Name of value in file (at least 3 chars!)
 * \param sequential 1 - do not reset file pointer to begin of file, 0 - set to beginning
 *        (if file is sequentially parsed this can be way faster! However, beware of multiple values per line, as whole line is read -
 *         best approach: 0 0 0 1 (not resetted only on last value of line) - and of yth, which is now
 *         counted from this unresetted position!)
 * \param xth Which among a number of parameters it is (in grid case it's row number as grid is read as a whole!)
 * \param yth In grid case specifying column number, otherwise the yth \a name matching line
 * \param type Type of the Parameter to be read
 * \param value address of the value to be read (must have been allocated)
 * \param repetition determines, if the keyword appears multiply in the config file, which repetition shall be parsed, i.e. 1 if not multiply
 * \param critical necessity of this keyword being specified (optional, critical)
 * \return 1 - found, 0 - not found
 * \note Routine is taken from the pcp project and hack-a-slack adapted to C++
 */
int ParseForParameter(const int verbose, struct ConfigFileBuffer * const FileBuffer, const char * const name, const int sequential, const int xth, const int yth, const int type, void * value, const int repetition, const int critical) {
  int i = 0;
  int j = 0;  // loop variables
  int length = 0;
  int maxlength = -1;
  int OldCurrentLine = FileBuffer->CurrentLine;
  char *dummy1 = NULL;
  char *dummy = NULL;    // pointers in the line that is read in per step

  //fprintf(stderr,"Parsing for %s\n",name);
  if (repetition == 0)
    //Error(SomeError, "ParseForParameter(): argument repetition must not be 0!");
    return 0;

  int line = 0; // marks line where parameter was found
  int found = (type >= grid) ? 0 : (-yth + 1);  // marks if yth parameter name was found
  while((found != repetition)) {
    dummy1 = dummy = NULL;
    do {
      dummy1 = FileBuffer->buffer[ FileBuffer->LineMapping[FileBuffer->CurrentLine++] ];
      if (FileBuffer->CurrentLine >= FileBuffer->NoLines) {
        if ((critical) && (found == 0)) {
          //Error(InitReading, name);
          fprintf(stderr,"Error:InitReading, critical %s not found\n", name);
          exit(255);
        } else {
          FileBuffer->CurrentLine = OldCurrentLine; // rewind to start position
          return 0;
        }
      }
      if (dummy1 == NULL) {
        if (verbose) fprintf(stderr,"Error reading line %i\n",line);
      } else {
        //fprintf(stderr,"Now parsing the line %i: %s\n", line, dummy1);
      }
      line++;
    } while (dummy1 != NULL && ((dummy1[0] == '#') || (dummy1[0] == '\0'))); // skip commentary and empty lines

    // Seek for possible end of keyword on line if given ...
    if (name != NULL) {
      dummy = strchr(dummy1,'\t');  // set dummy on first tab or space which ever's nearer
      if (dummy == NULL) {
        dummy = strchr(dummy1, ' ');  // if not found seek for space
        while ((dummy != NULL) && ((*dummy == '\t') || (*dummy == ' ')))    // skip some more tabs and spaces if necessary
          dummy++;
      }
      if (dummy == NULL) {
        dummy = strchr(dummy1, '\n'); // set on line end then (whole line = keyword)
        //fprintf(stderr,"Error: Cannot find tabs or spaces on line %i in search for %s\n", line, name);
        //Error(FileOpenParams, NULL);
      } else {
        //fprintf(stderr,"found tab at %i\n",(char *)dummy-(char *)dummy1);
      }
    } else dummy = dummy1;
    // ... and check if it is the keyword!
    //fprintf(stderr,"name %p, dummy %i/%c, dummy1 %i/%c, strlen(name) %i\n", &name, dummy, *dummy, dummy1, *dummy1, strlen(name));
    if ((name == NULL) || (((dummy-dummy1 >= 3) && (strncmp(dummy1, name, strlen(name)) == 0)) && ((unsigned int)(dummy-dummy1) == strlen(name)))) {
      found++; // found the parameter!
      //fprintf(stderr,"found %s at line %i between %i and %i\n", name, line, dummy1, dummy);

      if (found == repetition) {
        for (i=0;i<xth;i++) { // i = rows
          if (type >= grid) {
            // grid structure means that grid starts on the next line, not right after keyword
            dummy1 = dummy = NULL;
            do {
              dummy1 = FileBuffer->buffer[ FileBuffer->LineMapping[ FileBuffer->CurrentLine++] ];
              if (FileBuffer->CurrentLine >= FileBuffer->NoLines) {
                if ((critical) && (found == 0)) {
                  //Error(InitReading, name);
                  fprintf(stderr,"Error:InitReading, critical %s not found\n", name);
                  exit(255);
                } else {
                  FileBuffer->CurrentLine = OldCurrentLine; // rewind to start position
                  return 0;
                }
              }
              if (dummy1 == NULL) {
                if (verbose) fprintf(stderr,"Error reading line %i\n", line);
              } else {
                //fprintf(stderr,"Reading next line %i: %s\n", line, dummy1);
              }
              line++;
            } while ((dummy1 != NULL) && ((dummy1[0] == '#') || (dummy1[0] == '\n')));
            dummy = dummy1;
          } else { // simple int, strings or doubles start in the same line
            while ((*dummy == '\t') || (*dummy == ' '))  // skip interjacent tabs and spaces
              dummy++;
          }

          for (j=((type >= grid) ? 0 : yth-1);j<yth;j++) { // j = columns
            // check for lower triangular area and upper triangular area
            if ( ((i > j) && (type == upper_trigrid)) || ((j > i) && (type == lower_trigrid))) {
              *((double *)value) = 0.0;
              fprintf(stderr,"%f\t",*((double *)value));
              value = (void *)((long)value + sizeof(double));
              //value += sizeof(double);
            } else {
              // otherwise we must skip all interjacent tabs and spaces and find next value
              dummy1 = dummy;
              dummy = strchr(dummy1, '\t'); // seek for tab or space
              if (dummy == NULL)
                dummy = strchr(dummy1, ' ');  // if not found seek for space
              if (dummy == NULL) { // if still zero returned ...
                dummy = strchr(dummy1, '\n'); // ... at line end then
                if ((j < yth-1) && (type < 4)) {  // check if xth value or not yet
                  if (critical) {
                    if (verbose) fprintf(stderr,"Error: EoL at %i and still missing %i value(s) for parameter %s\n", line, yth-j, name);
                    //return 0;
                    exit(255);
                    //Error(FileOpenParams, NULL);
                  } else {
                    if (!sequential) { // here we need it!
                      FileBuffer->CurrentLine = OldCurrentLine; // rewind to start position
                    }
                    return 0;
                  }
                }
              } else {
                //fprintf(stderr,"found tab at %i\n",(char *)dummy-(char *)free_dummy);
              }
              if (*dummy1 == '#') {
                // found comment, skipping rest of line
                //if (verbose) fprintf(stderr,"Error: '#' at %i and still missing %i value(s) for parameter %s\n", line, yth-j, name);
                if (!sequential) { // here we need it!
                  FileBuffer->CurrentLine = OldCurrentLine; // rewind to start position
                }
                return 0;
              }
              //fprintf(stderr,"value from %i to %i\n",(char *)dummy1-(char *)free_dummy,(char *)dummy-(char *)free_dummy);
              switch(type) {
                case (row_int):
                  *((int *)value) = atoi(dummy1);
                  if ((verbose) && (i==0) && (j==0)) fprintf(stderr,"%s = ", name);
                  if (verbose) fprintf(stderr,"%i\t",*((int *)value));
                    value = (void *)((long)value + sizeof(int));
                    //value += sizeof(int);
                  break;
                case(row_double):
                case(grid):
                case(lower_trigrid):
                case(upper_trigrid):
                  *((double *)value) = atof(dummy1);
                  if ((verbose) && (i==0) && (j==0)) fprintf(stderr,"%s = ", name);
                  if (verbose) fprintf(stderr,"%lg\t",*((double *)value));
                  value = (void *)((long)value + sizeof(double));
                  //value += sizeof(double);
                  break;
                case(double_type):
                  *((double *)value) = atof(dummy1);
                  if ((verbose) && (i == xth-1)) fprintf(stderr,"%s = %lg\n", name, *((double *) value));
                  //value += sizeof(double);
                  break;
                case(int_type):
                  *((int *)value) = atoi(dummy1);
                  if ((verbose) && (i == xth-1)) fprintf(stderr,"%s = %i\n", name, *((int *) value));
                  //value += sizeof(int);
                  break;
                default:
                case(string_type):
                  if (value != NULL) {
                    //if (maxlength == -1) maxlength = strlen((char *)value); // get maximum size of string array
                    maxlength = MAXSTRINGSIZE;
                    length = maxlength > (dummy-dummy1) ? (dummy-dummy1) : maxlength; // cap at maximum
                    strncpy((char *)value, dummy1, length); // copy as much
                    ((char *)value)[length] = '\0'; // and set end marker
                    if ((verbose) && (i == xth-1)) fprintf(stderr,"%s is '%s' (%i chars)\n",name,((char *) value), length);
                    //value += sizeof(char);
                  } else {
                  }
                break;
              }
            }
            while (*dummy == '\t')
              dummy++;
          }
        }
      }
    }
  }
  if ((type >= row_int) && (verbose)) fprintf(stderr,"\n");
  if (!sequential) {
    FileBuffer->CurrentLine = OldCurrentLine; // rewind to start position
  }
  //fprintf(stderr, "End of Parsing\n\n");

  return (found); // true if found, false if not
}

/** Reading of Thermostat related values from parameter file.
 * \param *fb file buffer containing the config file
 */
void config::ParseThermostats(class ConfigFileBuffer * const fb)
{
  char * const thermo = new char[12];
  const int verbose = 0;

  // read desired Thermostat from file along with needed additional parameters
  if (ParseForParameter(verbose,fb,"Thermostat", 0, 1, 1, string_type, thermo, 1, optional)) {
    Thermostats->makeActive(thermo,fb);
  } else {
    if ((Thermostats->TargetTemp != 0))
      DoLog(2) && (Log() << Verbose(2) <<  "No thermostat chosen despite finite temperature MD, falling back to None." << endl);
    Thermostats->chooseNone();
  }
  delete[](thermo);
};

