/*
 * Interfragmenter.hpp
 *
 *  Created on: Jul 5, 2013
 *      Author: heber
 */

#ifndef INTERFRAGMENTER_HPP_
#define INTERFRAGMENTER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <list>
#include <map>
#include <vector>

#include "AtomIdSet.hpp"
#include "Fragmentation/HydrogenSaturation_enum.hpp"

class atom;
class KeySet;
class Graph;

/** This functor adds the union of certain fragments to a given set of fragments
 * (a Graph) by combining them depending on whether they are (not) bonded and
 * how far their centers are apart and which bond order they have.
 *
 * This is to allow calculation of interfragment energies. As fragments are
 * always of the same molecule, energies in between molecules so far are only
 * attained electrostratically, whereas dynamic correlation is totally missed.
 * Interfragments that are calculate with e.g. a sensible Post-HF method contain
 * dynamic correlation which can then be used for later potential fitting.
 */
class Interfragmenter
{
public:
  /** Constructor for class Interfragmenter.
   *
   * \param _TotalGraph Graph with all fragments to interrelate
   */
  Interfragmenter(Graph &_TotalGraph) :
    TotalGraph(_TotalGraph)
  {}

  /** Adds interrelated fragments to TotalGraph up to \a MaxOrder and \a Rcut.
   *
   * \param MaxOrder maximum order for fragments to interrelate
   * \param Rcut maximum distance to check for interrelatable fragments
   * \param treatment whether hydrogens are treated specially or not
   */
  void operator()(
      const size_t MaxOrder,
      const double Rcut,
      const enum HydrogenTreatment treatment);

private:
  /** Helper to translate a keyset into a set of atoms.
   *
   * \param keyset
   * \return vector of atom refs
   */
  std::vector<atom *> getAtomsFromKeySet(const KeySet &keyset) const;

  typedef std::list<const KeySet *> keysets_t;
  typedef std::map<const atom *, keysets_t > atomkeyset_t;

  /** Helper function to create a map of atoms and their fragments/keysets.
   *
   * \param _MaxOrder maximum order
   * \return map with atoms as a keys and fragments as values
   */
  atomkeyset_t getAtomKeySetMap(size_t _MaxOrder) const;

  typedef std::vector<const atom *> candidates_t;

  /** Helper function to get all atoms around a specific keyset not contained in
   * the same molecule.
   *
   * \param _atoms all atoms of a fragment
   * \param _Rcut desired distance cutoff
   * \param _treatment whether hydrogens are treated special or not
   */
  candidates_t getNeighborsOutsideMolecule(
      const AtomIdSet &_atoms,
      const double _Rcut,
      const enum HydrogenTreatment _treatment) const;

  /** Helper function to return a fragment/KeySet map specific to all candidates.
   *
   * \param _candidates all neighboring atoms around keyset
   * \param _atomkeyset map with all atoms and the KeySets they are contained in
   * \return specific fragment map
   */
  atomkeyset_t getCandidatesSpecificKeySetMap(
      const candidates_t &_candidates,
      const atomkeyset_t &_atomkeyset) const;

  /** For a given set of candidates atoms in \a _candidates and a \a _keyset
   * we combine each fragment from either atom and place it into internal
   * Graph.
   *
   * \param _MaxOrder maximum order
   * \param _candidates all atoms neighboring the current out outside of its molecule
   * \param _fragmentmap all keysets related to this atom
   * \param _keyset current keyset (used as base for creating inter-fragments)
   * \param _InterFragments container for all created inter-fragments
   * \param _counter counts added fragments
   */
  void combineFragments(
      const candidates_t &_candidates,
      atomkeyset_t &_fragmentmap,
      const KeySet &_keyset,
      Graph &_InterFragments,
      int &_counter);

private:
  Graph &TotalGraph;
};

#endif /* INTERFRAGMENTER_HPP_ */
