/*
 * Value_impl.hpp
 *
 *  Created on: Apr 13, 2012
 *      Author: ankele
 */

#ifndef VALUE_IMPL_HPP_
#define VALUE_IMPL_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <boost/any.hpp>

#include "CodePatterns/Assert.hpp"

#include "CodePatterns/Log.hpp"

#include "Validators/DummyValidator.hpp"
#include "Validators/DiscreteValidator.hpp"
#include "Validators/RangeValidator.hpp"
#include "ParameterExceptions.hpp"

// static member
template <class T> ConvertTo<T> Value<T>::Converter;

/** Constructor of class Value.
 */
template <class T>
Value<T>::Value() :
  ValueSet(false),
  validator(new DummyValidator<T>)
{}

/** Constructor of class Value with a validator.
 *
 * @param _validator general validator to use
 */
template <class T>
Value<T>::Value(const Validator<T> &_validator) :
  ValueSet(false),
  validator(_validator.clone())
{}

/** Constructor of class Value with a discrete validator.
 *
 * @param _ValidValues vector with all valid values
 */
template <class T>
Value<T>::Value(const std::vector<T> &_ValidValues) :
  ValueSet(false),
  validator(NULL)
{
  validator = new DiscreteValidator<T>(_ValidValues);
}

/** Constructor of class Value with a range validator.
 *
 * @param _ValidRange range of valid values
 */
template <class T>
Value<T>::Value(const range<T> &_ValidRange) :
  ValueSet(false),
  validator(NULL)
{
  validator = new RangeValidator<T>(_ValidRange);
}

/** Destructor of class Value.
 */
template <class T>
Value<T>::~Value()
{
  ASSERT(validator,
      "Value<T>::~Value() - validator missing.");
  delete(validator);
}

/** Checks whether \a _value is a valid value.
 * \param _value value to check for validity.
 * \return true - \a _value is valid, false - is not
 */
template <class T>
inline bool Value<T>::isValid(const T & _value) const throw(ParameterValidatorException)
{
  if (validator == NULL) throw ParameterValidatorException();
  return (*validator)(_value);
}

/** Compares this discrete value against another \a _instance.
 *
 * @param _instance other value to compare to
 * @return true - if value and valid ranges are the same, false - else
 */
template <class T>
bool Value<T>::operator==(const Value<T> &_instance) const throw(ParameterValidatorException)
{
  if (validator == NULL) throw ParameterValidatorException();
  if (_instance.validator == NULL) throw ParameterValidatorException();
  bool status = true;
  status = status && (*validator == *_instance.validator);
  status = status && (ValueSet == _instance.ValueSet);
  if (ValueSet && _instance.ValueSet)
    status = status && (value == _instance.value);
  return status;
}


/** Getter of value
 *
 * @return value
 */
template <class T>
inline const T & Value<T>::get() const throw(ParameterValueException)
{
  if (!isValid(value)) throw ParameterValueException();
  if (!ValueSet) throw ParameterValueException();
  return value;
}

/** Setter of value
 *
 * @param _value new value
 */
template <class T>
inline void Value<T>::set(const T & _value) throw(ParameterException)
{
  // any value may be set, this allows Actions to have invalid parameters
  // (e.g. because the given atom id does not yet exist) that are checked
  // on performCall()
//  if (!isValid(_value)) throw ParameterValueException();
  if (!ValueSet)
    ValueSet = true;
  value = _value;
}


/** Tests, if a value has been set
 *
 * @return true, if a value has been set
 */
template <class T>
inline bool Value<T>::isSet() const
{
  return ValueSet;
}



/** Checks whether \a _value is a valid value.
 * \param _value value to check for validity.
 * \return true - \a _value is valid, false - is not
 */
template <class T>
inline bool Value<T>::isValidAsString(const std::string &_value) const throw(ParameterValidatorException)
{
  const T castvalue = Converter(_value);
//  LOG(0, "Converted value reads " << castvalue <<".");
  return isValid(castvalue);
}

/** Getter of value, returning string.
 *
 * @return string value
 */
template <class T>
inline const std::string Value<T>::getAsString() const throw(ParameterValueException)
{
  return toString(get());
}

/** Setter of value for string
 *
 * @param _value string containing new value
 */
template <class T>
inline void Value<T>::setAsString(const std::string &_value) throw(ParameterException)
{
  const T castvalue = Converter(_value);
//  LOG(0, "Converted value reads " << castvalue <<".");
  set(castvalue);
//  LOG(0, "STATUS: Value is now set to " << value << ".");
}

/** Returns the validator as a const reference.
 *
 * @return the validator
 */
template <class T>
inline const Validator<T> &Value<T>::getValidator() const
{
  if (validator == NULL) throw ParameterValidatorException();
  return *validator;
}

/** Returns the validator.
 *
 * @return the validator
 */
template <class T>
inline Validator<T> &Value<T>::getValidator()
{
  if (validator == NULL) throw ParameterValidatorException();
  return *validator;
}



template <class T>
inline const range<T> & Value<T>::getValidRange() const throw(ParameterValidatorException)
{
  return dynamic_cast<const RangeValidator<T>&>(getValidator()).getValidRange();
}

/** Setter for the valid range.
 *
 * If value is invalid in new range, we throw ParameterValueException and set ValueSet to false.
 *
 * @param _range range (pair of values)
 */
template <class T>
inline void Value<T>::setValidRange(const range<T> &_range) throw(ParameterValueException)
{
  dynamic_cast<RangeValidator<T>&>(getValidator()).setValidRange(_range);
  if (ValueSet) {
      //std::cout << "Checking whether " << value << " is in range " << _range << "." << std::endl;
    if (!isValid(value)){
      //std::cout << "ValueSet to false." << std::endl;
      ValueSet = false;
      // have full check again in assert such that it appears in output, too
      throw ParameterValueException() << ParameterValidValues(toString(_range));
    }
  }
  //  LOG(0, "STATUS: Valid range is now " << ValidRange << ".");
}

template <class T>
inline void Value<T>::appendValidValue(const T &_value) throw(ParameterValidatorException)
{
  dynamic_cast<DiscreteValidator<T>&>(getValidator()).appendValidValue(_value);
}

template <class T>
inline const std::vector<T> &Value<T>::getValidValues() const throw(ParameterValidatorException)
{
  return dynamic_cast<const DiscreteValidator<T>&>(getValidator()).getValidValues();
}



#endif /* VALUE_IMPL_HPP_ */
