/** \file atom.cpp
 *
 * Function implementations for the class atom.
 *
 */

#include "atom.hpp"
#include "memoryallocator.hpp"

/************************************* Functions for class atom *************************************/

/** Constructor of class atom.
 */
atom::atom()
{
  father = this;  // generally, father is itself
  previous = NULL;
  next = NULL;
  Ancestor = NULL;
  type = NULL;
  sort = NULL;
  FixedIon = 0;
  GraphNr = -1;
  ComponentNr = NULL;
  IsCyclic = false;
  SeparationVertex = false;
  LowpointNr = -1;
  AdaptiveOrder = 0;
  MaxOrder = false;
  // set LCNode::Vector to our Vector
  node = &x;
};

/** Constructor of class atom.
 */
atom::atom(atom *pointer)
{
  Name = NULL;
  previous = NULL;
  next = NULL;
  father = pointer;  // generally, father is itself
  Ancestor = NULL;
  GraphNr = -1;
  ComponentNr = NULL;
  IsCyclic = false;
  SeparationVertex = false;
  LowpointNr = -1;
  AdaptiveOrder = 0;
  MaxOrder = false;
  type = pointer->type;  // copy element of atom
  x.CopyVector(&pointer->x); // copy coordination
  v.CopyVector(&pointer->v); // copy velocity
  FixedIon = pointer->FixedIon;
  nr = -1;
  sort = &nr;
  node = &x;
}


/** Destructor of class atom.
 */
atom::~atom()
{
  Free<int>(&ComponentNr, "atom::~atom: *ComponentNr");
  Free<char>(&Name, "atom::~atom: *Name");
  Trajectory.R.clear();
  Trajectory.U.clear();
  Trajectory.F.clear();
};


/** Climbs up the father list until NULL, last is returned.
 * \return true father, i.e. whose father points to itself, NULL if it could not be found or has none (added hydrogen)
 */
atom *atom::GetTrueFather()
{
  atom *walker = this;
  do {
    if (walker == walker->father) // top most father is the one that points on itself
      break;
    walker = walker->father;
  } while (walker != NULL);
  return walker;
};

/** Sets father to itself or its father in case of copying a molecule.
 */
void atom::CorrectFather()
{
  if (father->father == father)   // same atom in copy's father points to itself
    father = this;  // set father to itself (copy of a whole molecule)
  else
   father = father->father;  // set father to original's father

};

/** Check whether father is equal to given atom.
 * \param *ptr atom to compare father to
 * \param **res return value (only set if atom::father is equal to \a *ptr)
 */
void atom::EqualsFather ( atom *ptr, atom **res )
{
  if ( ptr == father )
    *res = this;
};

/** Checks whether atom is within the given box.
 * \param offset offset to box origin
 * \param *parallelepiped box matrix
 * \return true - is inside, false - is not
 */
bool atom::IsInParallelepiped(Vector offset, double *parallelepiped)
{
  return (node->IsInParallelepiped(offset, parallelepiped));
};

/** Output of a single atom.
 * \param ElementNo cardinal number of the element
 * \param AtomNo cardinal number among these atoms of the same element
 * \param *out stream to output to
 * \param *comment commentary after '#' sign
 */
bool atom::Output(ofstream *out, int ElementNo, int AtomNo, const char *comment) const
{
  if (out != NULL) {
    *out << "Ion_Type" << ElementNo << "_" << AtomNo << "\t"  << fixed << setprecision(9) << showpoint;
    *out << x.x[0] << "\t" << x.x[1] << "\t" << x.x[2];
    *out << "\t" << FixedIon;
    if (v.Norm() > MYEPSILON)
      *out << "\t" << scientific << setprecision(6) << v.x[0] << "\t" << v.x[1] << "\t" << v.x[2] << "\t";
    if (comment != NULL)
      *out << " # " << comment << endl;
    else
      *out << " # molecule nr " << nr << endl;
    return true;
  } else
    return false;
};
bool atom::Output(ofstream *out, int *ElementNo, int *AtomNo, const char *comment)
{
  AtomNo[type->Z]++;  // increment number
  if (out != NULL) {
    *out << "Ion_Type" << ElementNo[type->Z] << "_" << AtomNo[type->Z] << "\t"  << fixed << setprecision(9) << showpoint;
    *out << x.x[0] << "\t" << x.x[1] << "\t" << x.x[2];
    *out << "\t" << FixedIon;
    if (v.Norm() > MYEPSILON)
      *out << "\t" << scientific << setprecision(6) << v.x[0] << "\t" << v.x[1] << "\t" << v.x[2] << "\t";
    if (comment != NULL)
      *out << " # " << comment << endl;
    else
      *out << " # molecule nr " << nr << endl;
    return true;
  } else
    return false;
};

/** Output of a single atom as one lin in xyz file.
 * \param *out stream to output to
 */
bool atom::OutputXYZLine(ofstream *out) const
{
  if (out != NULL) {
    *out << type->symbol << "\t" << x.x[0] << "\t" << x.x[1] << "\t" << x.x[2] << "\t" << endl;
    return true;
  } else
    return false;
};

/** Output of a single atom as one lin in xyz file.
 * \param *out stream to output to
 */
bool atom::OutputTrajectory(ofstream *out, int *ElementNo, int *AtomNo, int step) const
{
  AtomNo[type->Z]++;
  if (out != NULL) {
    *out << "Ion_Type" << ElementNo[type->Z] << "_" << AtomNo[type->Z] << "\t"  << fixed << setprecision(9) << showpoint;
    *out << Trajectory.R.at(step).x[0] << "\t" << Trajectory.R.at(step).x[1] << "\t" << Trajectory.R.at(step).x[2];
    *out << "\t" << FixedIon;
    if (Trajectory.U.at(step).Norm() > MYEPSILON)
      *out << "\t" << scientific << setprecision(6) << Trajectory.U.at(step).x[0] << "\t" << Trajectory.U.at(step).x[1] << "\t" << Trajectory.U.at(step).x[2] << "\t";
    if (Trajectory.F.at(step).Norm() > MYEPSILON)
      *out << "\t" << scientific << setprecision(6) << Trajectory.F.at(step).x[0] << "\t" << Trajectory.F.at(step).x[1] << "\t" << Trajectory.F.at(step).x[2] << "\t";
    *out << "\t# Number in molecule " << nr << endl;
    return true;
  } else
    return false;
};

/** Output of a single atom as one lin in xyz file.
 * \param *out stream to output to
 */
bool atom::OutputTrajectoryXYZ(ofstream *out, int step) const
{
  if (out != NULL) {
    *out << type->symbol << "\t";
    *out << Trajectory.R.at(step).x[0] << "\t";
    *out << Trajectory.R.at(step).x[1] << "\t";
    *out << Trajectory.R.at(step).x[2] << endl;
    return true;
  } else
    return false;
};

ostream & operator << (ostream &ost, const atom &a)
{
  ost << "[" << a.Name << "|" << &a << "]";
  return ost;
};

ostream & atom::operator << (ostream &ost)
{
  ost << "[" << Name << "|" << this << "]";
  return ost;
};

/** Compares the indices of \a this atom with a given \a ptr.
 * \param ptr atom to compare index against
 * \return true - this one's is smaller, false - not
 */
bool atom::Compare(const atom &ptr)
{
  if (nr < ptr.nr)
    return true;
  else
    return false;
};

bool operator < (atom &a, atom &b)
{
  return a.Compare(b);
};

