/*
 * Registry_impl.hpp
 *
 *  Created on: Jul 28, 2010
 *      Author: heber
 */

#ifndef REGISTRY_IMPL_HPP_
#define REGISTRY_IMPL_HPP_

#include "Helpers/MemDebug.hpp"

#include "Patterns/Registry.hpp"
#include "Patterns/Singleton_impl.hpp"

#include "Helpers/Assert.hpp"
#include <iostream>

/** Constructor for class Registry.
 */
template <class T> Registry<T>::Registry()
{}

/** Destructor for class Registry.
 */
template <class T> Registry<T>::~Registry()
{}

/** Returns pointer to an instance named by \a name.
 * \param name name of instance
 * \return pointer to instance
 */
template <class T> T* Registry<T>::getByName(const std::string name) const
{
  typename std::map<const std::string,T*>::const_iterator iter;
  iter = InstanceMap.find(name);
  ASSERT(iter!=InstanceMap.end(),"Query for an instance "+name+" not stored in registry");
  return iter->second;
}

/** States whether instance is present or not.
 * \note This is needed as Registry<T>::getByName() ASSERT()s that instance is in std::map.
 * \param name name of instance
 * \return true - present, false - instance absent
 */
template <class T>bool Registry<T>::isPresentByName(const std::string name) const
{
  typename std::map<const std::string,T*>::const_iterator iter;
  iter = InstanceMap.find(name);
  return iter!=InstanceMap.end();
}

/** Registers an instance with the Registry.
 * \param *instance pointer to T.
 */
template <class T>void Registry<T>::registerInstance(T* instance){
  std::pair<typename std::map<const std::string,T*>::iterator,bool> ret;
  //std::cout << "Trying to register instance of type " << typeid(T).name() << " with name " << instance->getName() << "." << std::endl;
  ret = InstanceMap.insert(typename std::pair<const std::string,T*>(instance->getName(),instance));
  ASSERT(ret.second,"Two instances with the same name "+instance->getName()+" added to registry");
}

/** Unregisters an instance.
 * \param *instance pointer to instance.
 */
template <class T>void Registry<T>::unregisterInstance(T* instance){
  //std::cout << "Unregistering instance of type " << typeid(T).name() << " with name " << instance->getName() << "." << std::endl;
  InstanceMap.erase(instance->getName());
}

/** Removes every instance from the registry.
 */
template <class T>void Registry<T>::cleanup()
{
  typename std::map<const std::string,T*>::iterator iter;
  for(iter=InstanceMap.begin();iter!=InstanceMap.end();++iter) {
    delete iter->second;
  }
  InstanceMap.clear();
}


/** Returns an iterator pointing to the start of the std::map of instance's.
 * \return begin iterator
 */
template <class T>
typename std::map<const std::string,T*>::iterator Registry<T>::getBeginIter()
{
  return InstanceMap.begin();
}

/** Returns an iterator pointing to the end of the std::map of instance's.
 * \return end iterator
 */
template <class T>
typename std::map<const std::string,T*>::iterator Registry<T>::getEndIter()
{
  return InstanceMap.end();
}

/** Returns a const iterator pointing to the start of the std::map of instance's.
 * \return constant begin iterator
 */
template <class T>
typename std::map<const std::string,T*>::const_iterator Registry<T>::getBeginIter() const
{
  return InstanceMap.begin();
}

/** Returns a const iterator pointing to the end of the std::map of instance's.
 * \return constant end iterator
 */
template <class T>
typename std::map<const std::string,T*>::const_iterator Registry<T>::getEndIter() const
{
  return InstanceMap.end();
}

/** Prints the contents of the Registry \a &m to \a &ost.
 * \param &ost output stream
 * \param &m reference to Registry
 * \return reference to the above out stream for concatenation
 */
template <class T>
std::ostream& operator<<(std::ostream& ost, const Registry<T>& m)
{
  ost << "Registry contains:" << std::endl;
  for (typename std::map<const std::string,T*>::const_iterator iter = m.getBeginIter(); iter != m.getEndIter(); ++iter) {
    ost << "\t" << iter->first << " with pointer " << iter->second << std::endl;
  }
  return ost;
};

/**
 * This define allows simple instantiation of the necessary registryfunctions
 * at a chosen place.
 */
#define CONSTRUCT_REGISTRY(InstanceType) \
    template InstanceType* Registry<InstanceType>::getByName(const std::string) const; \
    template bool Registry<InstanceType>::isPresentByName(const std::string) const; \
    template void Registry<InstanceType>::registerInstance(InstanceType*); \
    template void Registry<InstanceType>::unregisterInstance(InstanceType*); \
    template std::map<const std::string,InstanceType*>::iterator Registry<InstanceType>::getBeginIter(); \
    template std::map<const std::string,InstanceType*>::const_iterator Registry<InstanceType>::getBeginIter() const; \
    template std::map<const std::string,InstanceType*>::iterator Registry<InstanceType>::getEndIter(); \
    template std::map<const std::string,InstanceType*>::const_iterator Registry<InstanceType>::getEndIter() const;


#endif /* REGISTRY_IMPL_HPP_ */
