/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserPcpUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserPcpUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "World.hpp"
#include "Atom/atom.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/PcpParser.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserPcpUnitTest );

static string waterPcp = "# ParallelCarParinello - main configuration file - created with molecuilder\n\
\n\
mainname\tpcp\t# programm name (for runtime files)\n\
defaultpath\not specified\t# where to put files during runtime\n\
pseudopotpath\not specified\t# where to find pseudopotentials\n\
\n\
ProcPEGamma\t8\t# for parallel computing: share constants\n\
ProcPEPsi\t1\t# for parallel computing: share wave functions\n\
DoOutVis\t0\t# Output data for OpenDX\n\
DoOutMes\t1\t# Output data for measurements\n\
DoOutOrbitals\t0\t# Output all Orbitals\n\
DoOutCurr\t0\t# Ouput current density for OpenDx\n\
DoOutNICS\t0\t# Output Nucleus independent current shieldings\n\
DoPerturbation\t0\t# Do perturbation calculate and determine susceptibility and shielding\n\
DoFullCurrent\t0\t# Do full perturbation\n\
DoConstrainedMD\t0\t# Do perform a constrained (>0, relating to current MD step) instead of unconstrained (0) MD\n\
Thermostat\tBerendsen\t2.5\t# Which Thermostat and its parameters to use in MD case.\n\
PcpWannier\t0\t# Put virtual centers at indivual orbits, all common, merged by variance, to grid point, to cell center\n\
SawtoothStart\t0.01\t# Absolute value for smooth transition at cell border \n\
VectorPlane\t0\t# Cut plane axis (x, y or z: 0,1,2) for two-dim current vector plot\n\
VectorCut\t0\t# Cut plane axis value\n\
AddGramSch\t1\t# Additional GramSchmidtOrtogonalization to be safe\n\
Seed\t1\t# initial value for random seed for Psi coefficients\n\
\n\
MaxOuterStep\t0\t# number of MolecularDynamics/Structure optimization steps\n\
Deltat\t0.01\t# time per MD step\n\
OutVisStep\t10\t# Output visual data every ...th step\n\
OutSrcStep\t5\t# Output \"restart\" data every ..th step\n\
TargetTemp\t0.000950045\t# Target temperature\n\
MaxPsiStep\t3\t# number of Minimisation steps per state (0 - default)\n\
EpsWannier\t1e-07\t# tolerance value for spread minimisation of orbitals\n\
# Values specifying when to stop\n\
MaxMinStep\t100\t# Maximum number of steps\n\
RelEpsTotalE\t1e-07\t# relative change in total energy\n\
RelEpsKineticE\t1e-05\t# relative change in kinetic energy\n\
MaxMinStopStep\t2\t# check every ..th steps\n\
MaxMinGapStopStep\t1\t# check every ..th steps\n\
\n\
# Values specifying when to stop for INIT, otherwise same as above\n\
MaxInitMinStep\t100\t# Maximum number of steps\n\
InitRelEpsTotalE\t1e-05\t# relative change in total energy\n\
InitRelEpsKineticE\t0.0001\t# relative change in kinetic energy\n\
InitMaxMinStopStep\t2\t# check every ..th steps\n\
InitMaxMinGapStopStep\t1\t# check every ..th steps\n\
\n\
BoxLength\t# (Length of a unit cell)\n\
20\n\
0\t20\n\
0\t0\t20\n\
\n\
ECut\t128\t# energy cutoff for discretization in Hartrees\n\
MaxLevel\t5\t# number of different levels in the code, >=2\n\
Level0Factor\t2\t# factor by which node number increases from S to 0 level\n\
RiemannTensor\t0\t# (Use metric)\n\
PsiType\t0\t# 0 - doubly occupied, 1 - SpinUp,SpinDown\n\
MaxPsiDouble\t2\t# here: specifying both maximum number of SpinUp- and -Down-states\n\
PsiMaxNoUp\t2\t# here: specifying maximum number of SpinUp-states\n\
PsiMaxNoDown\t2\t# here: specifying maximum number of SpinDown-states\n\
AddPsis\t0\t# Additional unoccupied Psis for bandgap determination\n\
\n\
RCut\t20\t# R-cut for the ewald summation\n\
StructOpt\t0\t# Do structure optimization beforehand\n\
IsAngstroem\t1\t# 0 - Bohr, 1 - Angstroem\n\
RelativeCoord\t0\t# whether ion coordinates are relative (1) or absolute (0)\n\
MaxTypes\t2\t# maximum number of different ion types\n\
\n\
# Ion type data (PP = PseudoPotential, Z = atomic number)\n\
#Ion_TypeNr.\tAmount\tZ\tRGauss\tL_Max(PP)L_Loc(PP)IonMass\t# chemical name, symbol\n\
Ion_Type1\t2\t1\t1.0\t3\t3\t1.008\tHydrogen\tH\n\
Ion_Type2\t1\t8\t1.0\t3\t3\t15.999\tOxygen\tO\n\
#Ion_TypeNr._Nr.R[0]\tR[1]\tR[2]\tMoveType (0 MoveIon, 1 FixedIon)\n\
Ion_Type2_1\t1.000000000\t0.000000000\t0.000000000\t0 # molecule nr 0\n\
Ion_Type1_1\t0.758602\t0.000000000\t0.504284\t0 # molecule nr 1\n\
Ion_Type1_2\t0.758602\t0.000000000\t-0.504284\t0 # molecule nr 2\n";

void ParserPcpUnitTest::setUp() {
  World::getInstance();

  parser = new FormatParser<pcp>();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserPcpUnitTest::tearDown() 
{
  delete parser;
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserPcpUnitTest::readwritePcpTest() {
  stringstream input(waterPcp);
  parser->load(&input);
  input.clear();

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  // check that equality function is ok
  CPPUNIT_ASSERT(*parser == *parser);

  stringstream output;
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  parser->save(&output, atoms);

  input << output.str();
  FormatParser<pcp>* parser2 = new FormatParser<pcp>();
  parser2->load(&input);

  CPPUNIT_ASSERT_EQUAL(6, World::getInstance().numAtoms());

  CPPUNIT_ASSERT(*parser == *parser2);

  delete parser2;
}
