/*
 * Particle.hpp
 *
 *  Created on: May 13, 2013
 *      Author: heber
 */

#ifndef PARTICLE_HPP_
#define PARTICLE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <string>
#include <vector>

#include "types.hpp"

class periodentafel;

class Particle
{
public:
  /** Constructor for class Particle.
   *
   * Chooses token as combination of element's symbol and a number such that name is
   * unique with registry.
   *
   * \param _periode reference to periodentafel for looking up element nr
   * \param _number atomic number of particle's element
   */
  Particle(const periodentafel &_periode, const atomicNumber_t &_number);

  /** Constructor for class Particle.
   *
   * \param _periode reference to periodentafel for looking up element nr
   * \param _token unique token/name of this particle
   * \param _number atomic number of particle's element
   */
  Particle(const periodentafel &_periode, const std::string &_token, const atomicNumber_t &_number);

  /** Destructor for class Particle.
   *
   */
  ~Particle() {}

  /** Getter for the name of this Particle.
   *
   * This function is required such that Particle's can be stored in a registry.
   *
   * \return name of particle
   */
  const std::string& getName() const
  { return name; }

  /** Returns the name of the element.
   *
   * \return name of the particle's element
   */
  std::string getElement() const;

  /** Print parameters to given stream \a ost.
   *
   * These are virtual functions to allow for overriding and hence
   * changing the default behavior.
   *
   * @param ost stream to print to
   */
  void stream_to(std::ostream &ost) const;

  /** Parse parameters from given stream \a ist.
   *
   * These are virtual functions to allow for overriding and hence
   * changing the default behavior.
   *
   * @param ist stream to parse from
   */
  void stream_from(std::istream &ist);

private:
  //!> grant factory access to default cstor.
  friend class ParticleFactory;

  /** Default constructor for class Particle.
   *
   * \warning default constructor is private to prevent Particle without a
   * unique name.
   *
   * \param _periode reference to periodentafel for looking up element nr
   */
  Particle(const periodentafel &_periode);

  /** Helper function to find index to a parameter name.
   *
   * \param name name of parameter to look up index for
   * \return index in ParameterNames or -1 if not found
   */
  size_t lookupParameterName(const std::string &name) const;

  /** Setter for the particle's element.
   *
   * \param element_name name of particle's element, must be known to periodentafel.
   */
  void setElement(const std::string &element_name);

public:

  /** Finds the next free token in the registry for the given element.
   *
   * \param _periode reference to periodentafel for looking up element nr
   * \param _number atomic number of particle's element
   * \return unique token for this element
   */
  static std::string findFreeName(
      const periodentafel &_periode,
      const atomicNumber_t &_number);

  //!> token/name of this particle
  const std::string name;

  //!> reference to periodentafel to look up elements
  const periodentafel &periode;

  //!> partial charge of this particle
  double charge;
  //!> (effective) mass of this particle
  double mass;
  //!> (effective) degrees of freedom
  unsigned int dof;
  //!> atomic number of the particle's element
  atomicNumber_t atomic_number;

  // the following variables are due to convention in .potentials file
  double sigma;
  double epsilon;
  double sigma_14;
  double epsilon_14;

  enum parameters_t {
    e_particle_type,
    e_element_name,
    e_sigma,
    e_epsilon,
    e_sigma_14,
    e_epsilon_14,
    e_mass,
    e_free,
    e_charge,
    MAXPARAMETERS
  };

  static const std::vector<std::string> ParameterNames;
};

/** Output stream operator for class Particle.
 *
 * \param ost output stream
 * \param p Particle instance
 */
std::ostream& operator<<(std::ostream &ost, const Particle &p);

#endif /* PARTICLE_HPP_ */
