/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Particle.cpp
 *
 *  Created on: May 13, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Particle.hpp"

#include <boost/assign.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/tokenizer.hpp>
#include <iostream>
#include <iterator>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Potentials/Exceptions.hpp"
#include "Potentials/Particles/ParticleRegistry.hpp"

using namespace boost::assign;

std::vector<std::string> getParameterNames()
{
  std::vector<std::string> tempnames;
  tempnames += "particle_type", "element_name", "sigma", "epsilon", "sigma14", "epsilon14", "mass", "free", "charge";
  return tempnames;
}

const std::vector<std::string> Particle::ParameterNames = getParameterNames();

Particle::Particle(
    const periodentafel &_periode,
    const std::string &_token,
    const atomicNumber_t &_number) :
    name(_token),
    periode(_periode),
    charge(0.),
    mass(0.),
    dof(3),
    atomic_number(_number),
    sigma(0.),
    epsilon(0.),
    sigma_14(0.),
    epsilon_14(0.)
{
  ParticleRegistry::getInstance().registerInstance(this);
}

std::string Particle::findFreeName(
    const periodentafel &_periode,
    const atomicNumber_t &_number)
{
  const element *e = _periode.FindElement(_number);
  ASSERT (e != NULL,
      "Particle::findFreeName() - cannot find element with number "
      +toString(_number)+".");
  bool FoundFlag = false;
  size_t index = 1;
  std::string returnname;
  while(!FoundFlag) {
    returnname = e->getSymbol()+toString(index++);
    FoundFlag = !ParticleRegistry::getInstance().isPresentByName(returnname);
  }
  return returnname;
}

Particle::Particle(
    const periodentafel &_periode,
    const atomicNumber_t &_number) :
    name(findFreeName(_periode, _number)),
    periode(_periode),
    charge(0.),
    mass(0.),
    dof(3),
    atomic_number(_number),
    sigma(0.),
    epsilon(0.),
    sigma_14(0.),
    epsilon_14(0.)
{
  ParticleRegistry::getInstance().registerInstance(this);
}

Particle::Particle(const periodentafel &_periode) :
    periode(_periode),
    charge(0.),
    mass(0.),
    dof(3),
    atomic_number(-1),
    sigma(0.),
    epsilon(0.),
    sigma_14(0.),
    epsilon_14(0.)
{
  // not registered as it does not have a name yet
}

void Particle::stream_to(std::ostream &ost) const
{
  // check stream
  if (ost.bad())
    throw SerializerStreamException();

  /// print parameter key
  ost << "\tparticle:";
  /// print name and values
  ost << "\tparticle_type=" << getName();
  ost << ",\telement_name=" << getElement();
  ost << ",\tsigma=" << sigma;
  ost << ",\tepsilon=" << epsilon;
  ost << ",\tsigma14=" << sigma_14;
  ost << ",\tepsilon14=" << epsilon_14;
  ost << ",\tmass=" << mass;
  ost << ",\tfree=" << dof;
  ost << ",\tcharge=" << charge;
  ost << ";";
}

size_t Particle::lookupParameterName(const std::string &name) const
{
  std::vector<std::string>::const_iterator iter =
      std::find(ParameterNames.begin(), ParameterNames.end(), name);
  if (iter == ParameterNames.end())
    return -1;
  else
    return std::distance(ParameterNames.begin(), iter);
}

void Particle::stream_from(std::istream &ist)
{
  // check stream
  if (ist.bad())
    throw SerializerStreamException();

  // read in full line
  std::string linestring;
  getline(ist, linestring);
  const std::string whitespace(" \t");
  const size_t strBegin = linestring.find_first_not_of(whitespace);
  const size_t colonpos = linestring.find(":");
  if ((strBegin == std::string::npos) || (colonpos == std::string::npos) ||
      (linestring.substr(strBegin, colonpos-strBegin) != std::string("particle")))
    throw SerializerMissingValueException()
        << SerializerKey(std::string("particle"));

  // tokenize by ","
  typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
  boost::char_separator<char> pairsep(",\t ;");
  std::string remainderstring(linestring.substr(colonpos+1));
  tokenizer tokens(remainderstring, pairsep); //skip colon

  // step through each token
  for (tokenizer::iterator tok_iter = tokens.begin();
    tok_iter != tokens.end(); ++tok_iter) {
    const std::string &keyvalue = *tok_iter;
    const size_t equalitypos = keyvalue.find("=");
    const std::string key = keyvalue.substr(0,equalitypos);
    const std::string value = keyvalue.substr(equalitypos+1);

    const size_t index = lookupParameterName(key);

    if (equalitypos == std::string::npos)
      throw SerializerMissingValueException() << SerializerKey(key);

    switch((parameters_t)index) {
    case e_particle_type:
      const_cast<std::string &>(name) = value;
      break;
    case e_element_name:
      setElement(value);
      break;
    case e_sigma:
      sigma = boost::lexical_cast<double>(value);
      break;
    case e_epsilon:
      epsilon = boost::lexical_cast<double>(value);
      break;
    case e_sigma_14:
      sigma_14 = boost::lexical_cast<double>(value);
      break;
    case e_epsilon_14:
      epsilon_14 = boost::lexical_cast<double>(value);
      break;
    case e_mass:
      mass = boost::lexical_cast<double>(value);
      break;
    case e_free:
      dof = boost::lexical_cast<unsigned int>(value);
      break;
    case e_charge:
      charge = boost::lexical_cast<double>(value);
      break;
    default:
      throw SerializerMissingValueException() << SerializerKey(key);
      break;
    }
  }
}

std::ostream& operator<<(std::ostream &ost, const Particle &p)
{
  p.stream_to(ost);
  return ost;
}

std::string Particle::getElement() const
{
  std::string returnstring;
  const element *e = periode.FindElement(atomic_number);
  if (e != NULL)
    returnstring = e->getSymbol();

  return returnstring;
}

void Particle::setElement(const std::string &element_name)
{
  const element *e = periode.FindElement(element_name);
  if (e != NULL)
    atomic_number = e->getAtomicNumber();
}
