/*
 * PairPotential_Morse.hpp
 *
 *  Created on: Oct 03, 2012
 *      Author: heber
 */

#ifndef PAIRPOTENTIAL_MORSE_HPP_
#define PAIRPOTENTIAL_MORSE_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits>

#include "CodePatterns/Assert.hpp"

#include "Potentials/EmpiricalPotential.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

/** This is the implementation of a harmonic pair potential.
 *
 * This evaluates \f$ k \cdot (r -r_0)^2 \f$.
 *
 */
class PairPotential_Morse : virtual public EmpiricalPotential, virtual public FunctionModel
{
  //!> grant unit test access to internal parts
  friend class PairPotential_MorseTest;
  // some repeated typedefs to avoid ambiguities
  typedef FunctionModel::arguments_t arguments_t;
  typedef FunctionModel::result_t result_t;
  typedef FunctionModel::results_t results_t;
  typedef EmpiricalPotential::derivative_components_t derivative_components_t;
  typedef FunctionModel::parameters_t parameters_t;
public:
  PairPotential_Morse();
  PairPotential_Morse(
      const double _spring_constant,
      const double _equilibrium_distance,
      const double _dissociation_energy,
      const double _energy_offset);
  virtual ~PairPotential_Morse() {}

  /** Setter for parameters as required by FunctionModel interface.
   *
   * \param _params given set of parameters
   */
  void setParameters(const parameters_t &_params);

  /** Getter for parameters as required by FunctionModel interface.
   *
   * \return set of parameters
   */
  parameters_t getParameters() const
  {
    return params;
  }

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  size_t getParameterDimension() const
  {
    return 4;
  }

  /** Evaluates the harmonic potential function for the given arguments.
   *
   * @param arguments single distance
   * @return value of the potential function
   */
  results_t operator()(const arguments_t &arguments) const;

  /** Evaluates the derivative of the potential function.
   *
   * @param arguments single distance
   * @return vector with derivative with respect to the input degrees of freedom
   */
  derivative_components_t derivative(const arguments_t &arguments) const;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param arguments set of arguments as input variables to the function
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  results_t parameter_derivative(const arguments_t &arguments, const size_t index) const;

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  bool isBoxConstraint() const {
    return true;
  }

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  parameters_t getLowerBoxConstraints() const {
    parameters_t lowerbound(getParameterDimension(), -std::numeric_limits<double>::max());
    lowerbound[equilibrium_distance] = 0.;
    return lowerbound;
  }

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  parameters_t getUpperBoxConstraints() const {
    return parameters_t(getParameterDimension(), std::numeric_limits<double>::max());
  }

  enum parameter_enum_t {
    spring_constant=0,
    equilibrium_distance=1,
    dissociation_energy=2,
    energy_offset=3,
    MAXPARAMS
  };

private:
  //!> parameter vector with parameters as in enum parameter_enum_t
  parameters_t params;
};

#endif /* PAIRPOTENTIAL_MORSE_HPP_ */
