/*
 * RandomNumberGenerator_Encapsulation.hpp
 *
 *  Created on: Dec 31, 2010
 *      Author: heber
 */

#ifndef RANDOMNUMBERGENERATOR_ENCAPSULATION_HPP_
#define RANDOMNUMBERGENERATOR_ENCAPSULATION_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/nondet_random.hpp>
#include <boost/random.hpp>
#include <boost/random/bernoulli_distribution.hpp>
#include <boost/random/binomial_distribution.hpp>
#include <boost/random/cauchy_distribution.hpp>
#include <boost/random/exponential_distribution.hpp>
#include <boost/random/gamma_distribution.hpp>
#include <boost/random/geometric_distribution.hpp>
#include <boost/random/linear_congruential.hpp>
#include <boost/random/lognormal_distribution.hpp>
#include <boost/random/normal_distribution.hpp>
#include <boost/random/poisson_distribution.hpp>
#include <boost/random/triangle_distribution.hpp>
#include <boost/random/uniform_01.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/random/uniform_on_sphere.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/random/uniform_smallint.hpp>
#include <boost/random/additive_combine.hpp>
#include <boost/random/discard_block.hpp>
#include <boost/random/inversive_congruential.hpp>
#include <boost/random/lagged_fibonacci.hpp>
#include <boost/random/linear_congruential.hpp>
#include <boost/random/linear_feedback_shift.hpp>
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/random_number_generator.hpp>
#include <boost/random/ranlux.hpp>
#include <boost/random/shuffle_output.hpp>
#include <boost/random/subtract_with_carry.hpp>
#include <boost/random/xor_combine.hpp>
#include <boost/random/variate_generator.hpp>

#include <typeinfo>

#include "CodePatterns/Clone.hpp"
#include "RandomNumberDistribution_Encapsulation.hpp"
#include "RandomNumberDistributionFactory.hpp"
#include "RandomNumberEngine_Encapsulation.hpp"
#include "RandomNumberEngineFactory.hpp"
#include "RandomNumberGenerator.hpp"

class RandomNumberGeneratorFactory;

/** Template class that encapsulates the random number generators from
 *  random::boost.
 *
 * We inherit the interface RandomNumberGenerator such that all are
 * accessible in the same way (i.e. RandomNumberGeneratorFactory will
 * spit out only references to RandomNumberGenerator).
 *
 * Note that we always returns double values although the distribution
 * might be integer or even a discrete distribution of integers.
 *
 * We need two template parameters:
 * -# the engine - generates uniform random numbers
 * -# the distribution - transforms uniform into a desired distribution
 */
template <class engine, class distribution>
class RandomNumberGenerator_Encapsulation :
  public RandomNumberGenerator,
  public Clone<RandomNumberGenerator>
{
  /**
   * Factory is friend such that it can access private cstor when filling its
   * table
   */
  friend class RandomNumberGeneratorFactory;

public:
  /** obtain a random number via generator and the specific distribution.
   *
   */
  double operator()() const {
    return (*randomgenerator)();
  }

  /** Set the generator's seed.
   *
   * @param _seed seed to set to
   */
  void seed(unsigned int _seed) {
    engine_type->seed(_seed);
  }

  /** Getter for the type name of the internal engine.
   *
   */
  std::string EngineName() {
    return engine_type->name();
  }

  /** Getter for the type name of the internal distribution.
   *
   */
  std::string DistributionName() {
    return distribution_type->name();
  }

  /** Clones the current instance and returns pointer.
   *
   * @return pointer to cloned instance
   */
  RandomNumberGenerator* clone() const
  {
    RandomNumberGenerator_Encapsulation<engine, distribution> *MyClone = NULL;

    // sadly (due to construction of variate_generator without any abstract
    // base class) we need RTTI here ...
    RandomNumberEngine *engine_clone =
        RandomNumberEngineFactory::getInstance().getProduct(typeid(engine));
    RandomNumberDistribution *distribution_clone =
        RandomNumberDistributionFactory::getInstance().getProduct(typeid(distribution));

    MyClone = new RandomNumberGenerator_Encapsulation<engine, distribution>(
        engine_clone,
        distribution_clone
        );

    return MyClone;
  }

protected:
  /** Constructor that instantiates a specific random number generator and
   * distribution.
   *
   * This is one is supposed to create the prototypes. Hence, must only be
   * called from the factory.
   */
  RandomNumberGenerator_Encapsulation() :
    engine_type(NULL),
    distribution_type(NULL),
    randomgenerator(NULL)
  {
    engine_type = RandomNumberEngineFactory::getInstance().getProduct(typeid(engine));
    distribution_type = RandomNumberDistributionFactory::getInstance().getProduct(typeid(distribution));
    randomgenerator = new boost::variate_generator<engine, distribution>(
        (dynamic_cast<RandomNumberEngine_Encapsulation<engine> *>(engine_type))
        ->getEngine(),
        (dynamic_cast<RandomNumberDistribution_Encapsulation<distribution> *>(distribution_type))
        ->getDistribution());
  }

  /** Constructor that instantiates a specific random number generator and
   * distribution.
   * @param _engine_type instance of the desired generator
   * @param _distribution_type instance of the desired distribution
   */
  RandomNumberGenerator_Encapsulation(
      RandomNumberEngine *_engine_type,
      RandomNumberDistribution *_distribution_type
      ) :
    engine_type(_engine_type),
    distribution_type(_distribution_type),
    randomgenerator(
        new boost::variate_generator<engine, distribution> (
            (dynamic_cast<RandomNumberEngine_Encapsulation<engine> *>(engine_type))
            ->getEngine(),
            (dynamic_cast<RandomNumberDistribution_Encapsulation<distribution> *>(distribution_type))
            ->getDistribution())
    )
  {}

  /** Destructor of the class.
   *
   */
  virtual ~RandomNumberGenerator_Encapsulation()
  {
    // NULL pointer may be deleted
    delete engine_type;
    delete distribution_type;
    delete randomgenerator;
  }
protected:
  RandomNumberEngine *engine_type;
  RandomNumberDistribution *distribution_type;\
private:
  mutable boost::variate_generator<engine, distribution> *randomgenerator;
};

#endif /* RANDOMNUMBERGENERATOR_ENCAPSULATION_HPP_ */
