/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RandomNumberGeneratorFactory.cpp
 *
 *  Created on: Dec 31, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <utility>
#include "CodePatterns/Singleton_impl.hpp"
#include "CodePatterns/Assert.hpp"

#include "TemplatePowerSetGenerator.hpp"
#include "EmptyPrototypeTable.hpp"

#include <boost/preprocessor/facilities/empty.hpp>
#include <boost/preprocessor/punctuation/paren.hpp>
#include <boost/preprocessor/seq/for_each_product.hpp>
#include <boost/preprocessor/facilities/identity.hpp>
#include <boost/preprocessor/facilities/expand.hpp>
#include <boost/preprocessor/seq/seq.hpp>

#include "RandomNumberGenerator_Encapsulation.hpp"

#include "RandomNumberGeneratorFactory.hpp"
#include "RandomNumberGeneratorFactory.def"

RandomNumberDistributionFactory::TypeList RandomNumberGeneratorFactory::distribution = (RandomNumberDistributionFactory::TypeList) 0;
RandomNumberEngineFactory::TypeList RandomNumberGeneratorFactory::engine = (RandomNumberEngineFactory::TypeList) 0;
RandomNumberGeneratorFactory::EngineDistributionTable RandomNumberGeneratorFactory::GeneratorPrototypeTable;

RandomNumberGeneratorFactory::RandomNumberGeneratorFactory()
{
	FillPrototypeTable();
}

RandomNumberGeneratorFactory::~RandomNumberGeneratorFactory()
{
  // clear out factories map to allow boost::shared_ptr to do their work (i.e. to release mem)
  // this is necessary as factories is a object
  for (EngineDistributionTable::iterator iter = GeneratorPrototypeTable.begin();
      !GeneratorPrototypeTable.empty();
      iter = GeneratorPrototypeTable.begin()) {
    EmptyPrototypeTable< std::map<RandomNumberDistributionFactory::TypeList,Clone<RandomNumberGenerator> *> > (iter->second);
    GeneratorPrototypeTable.erase(iter);
  }
  GeneratorPrototypeTable.clear();
}

void RandomNumberGeneratorFactory::FillPrototypeTable()
{
  // fill GeneratorPrototypeTable
#define SequenceElementizer(z,data) (data)

#define distributionengine_seqseq \
  BOOST_PP_SEQ_FOR_EACH_PRODUCT(SequenceElementizer, (engine_seq)(distribution_seq))
#define distributionengine_seqseq_a \
  BOOST_PP_SEQ_FOR_EACH_PRODUCT(SequenceElementizer, (engine_seq_a)(distribution_seq))

#define suffixseq ()(<>)
#define tableseq (*EnginePrototypeTable)(*DistributionPrototypeTable)
#define name_spaces_seq (RandomNumberEngineFactory::)(RandomNumberDistributionFactory::)

#define size_tupels BOOST_PP_SEQ_SIZE(tableseq)

#define TableItemPrinter(z,n,sequence) \
  [ \
  BOOST_PP_SEQ_ELEM(n,name_spaces_seq) \
  BOOST_PP_SEQ_ELEM(n,sequence) \
  ]

#define TemplateItemPrinter(z,n,sequence) \
  BOOST_PP_COMMA_IF(n) \
  boost:: \
  BOOST_PP_SEQ_ELEM(n,sequence) \
  BOOST_PP_SEQ_ELEM(n,suffixseq)

#define ParamItemPrinter(z,n,sequence)

#define BOOST_PP_LOCAL_MACRO(n) seqitems_as_enum_key_multidimmap(~, n, size_tupels, distributionengine_seqseq, GeneratorPrototypeTable, TableItemPrinter, new RandomNumberGenerator_Encapsulation , TemplateItemPrinter, ParamItemPrinter)
#define BOOST_PP_LOCAL_LIMITS  (0, BOOST_PP_SEQ_SIZE(distributionengine_seqseq)-1 )
#include BOOST_PP_LOCAL_ITERATE()

#define BOOST_PP_LOCAL_MACRO(n) seqitems_as_enum_key_multidimmap(~, n, size_tupels, distributionengine_seqseq_a, GeneratorPrototypeTable, TableItemPrinter, new RandomNumberGenerator_Encapsulation , TemplateItemPrinter, ParamItemPrinter)
#define BOOST_PP_LOCAL_LIMITS  (0, BOOST_PP_SEQ_SIZE(distributionengine_seqseq_a)-1 )
#include BOOST_PP_LOCAL_ITERATE()
}

RandomNumberGenerator* RandomNumberGeneratorFactory::makeRandomNumberGenerator() const
{
  // Instantiate and return (implicitly creates a copy of the stored prototype)
  return (GeneratorPrototypeTable[engine][distribution]->clone());
}

RandomNumberGenerator* RandomNumberGeneratorFactory::makeRandomNumberGenerator(
    std::string engine_type,
    std::string distribution_type
    ) const
{
  RandomNumberEngineFactory::TypeList eng_type;
  RandomNumberDistributionFactory::TypeList dis_type;

  // Instantiate and return (implicitly creates a copy of the stored prototype)
  if (!engine_type.empty()) {
    eng_type = RandomNumberEngineFactory::getInstance().getEnum(engine_type);
  } else
    eng_type = engine;
  if (!distribution_type.empty()) {
    dis_type = RandomNumberDistributionFactory::getInstance().getEnum(distribution_type);
  } else
    dis_type = distribution;
  return (GeneratorPrototypeTable[ eng_type ][ dis_type ]->clone());
}

void RandomNumberGeneratorFactory::setEngine(std::string engine_type)
{
  engine = RandomNumberEngineFactory::getInstance().getEnum(engine_type);
}

const std::string &RandomNumberGeneratorFactory::getEngineName() const
{
  return RandomNumberEngineFactory::getInstance().getName(engine);
}

void RandomNumberGeneratorFactory::setDistribution(std::string distribution_type)
{
  distribution = RandomNumberDistributionFactory::getInstance().getEnum(distribution_type);
}

const std::string &RandomNumberGeneratorFactory::getDistributionName() const
{
  return RandomNumberDistributionFactory::getInstance().getName(distribution);
}

CONSTRUCT_SINGLETON(RandomNumberGeneratorFactory)

#include "RandomNumberGeneratorFactory.undef"

