/*
 * dirichlet_cs.cpp
 *
 *  Created on: 25.01.2011
 *      Author: Julian Iseringhausen
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/test/unit_test.hpp>
#include <boost/test/floating_point_comparison.hpp>

#include "base/factory.hpp"
#include "base/math.hpp"
#include "base/vector.hpp"
#include "comm/comm_serial.hpp"
#include "level/level_operator_cs.hpp"
#include "level/level_operator.hpp"
#include "level/stencils.hpp"
#include "samples/discretization_poisson_fd.hpp"
#include "samples/techniques.hpp"
#include "smoother/gsrb.hpp"
#ifdef HAVE_LAPACK
#include "solver/dgesv.hpp"
#endif
#include "solver/givens.hpp"
#include "solver/solver_regular.hpp"
#include "mg.hpp"

#include "interface_sinus.hpp"

using namespace VMG;

const vmg_float sine_factor = static_cast<vmg_float>(2.0 * Math::pi);

struct LibraryDirichletCSFixture
{
  LibraryDirichletCSFixture()
  {
    Factory& factory = MG::GetFactory();

    Comm *comm = new CommSerial(Boundary(Dirichlet, Dirichlet, Dirichlet));
    comm->Register("COMM");

    Interface* interface = new VMGInterfaces::InterfaceSinus(sine_factor, comm->BoundaryConditions(), 2, 6, 0.0, 1.0);
    MG::SetInterface(interface, comm);

    Discretization* discretization = new DiscretizationPoissonFD(2);
    discretization->Register("DISCRETIZATION");

    LevelOperator* lop = new LevelOperatorCS(Stencils::RestrictionFullWeight, Stencils::InterpolationTrilinear);
    lop->Register("LEVEL_OPERATOR");

    Smoother* smoother = new GaussSeidelRB();
    smoother->Register("SMOOTHER");

#ifdef HAVE_LAPACK
    Solver* solver = new DGESV<SolverRegular>();
#else
    Solver* solver = new Givens<SolverRegular>();
#endif
    solver->Register("SOLVER");

    Techniques::SetCorrectionSchemeDirichlet(interface->MinLevel(), interface->MaxLevel(), 2);

    factory.RegisterObjectStorage("PRESMOOTHSTEPS", 3);
    factory.RegisterObjectStorage("POSTSMOOTHSTEPS", 3);
    factory.RegisterObjectStorage("PRECISION", 1e-10);
    factory.RegisterObjectStorage("MAX_ITERATION", 7);

    MG::IsInitialized();
  }

  ~LibraryDirichletCSFixture()
  {
    MG::Destroy();
  }
};

BOOST_FIXTURE_TEST_CASE(LibraryDirichletCSTest, LibraryDirichletCSFixture)
{
  MG::Solve();

  double res_init = MG::GetFactory().Get("INITIAL_RESIDUAL")->Cast< ObjectStorage<double> >()->Val();
  double res = MG::GetComm()->ComputeResidualNorm(*MG::GetSol(), *MG::GetRhs());

  BOOST_CHECK_SMALL(res/res_init, 1e-10);
}
