/*
 * VMGJob.hpp
 *
 *  Created on: Jul 12, 2012
 *      Author: heber
 */

#ifndef VMGJOB_HPP_
#define VMGJOB_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "boost/serialization/export.hpp"
#include "boost/serialization/vector.hpp"

#include "JobMarket/Jobs/FragmentJob.hpp"
#include "Jobs/Grid/SamplingGrid.hpp"
#include "Jobs/VMGData.hpp"

#include <vector>

/** This class encapsulates a VMG Job.
 *
 *  VMGJob calculates the long-range contribution that is missed out so far in the
 *  MPQCJob where the SCF cycle is calculated. To this end
 *
 */
class VMGJob : public FragmentJob
{
public:
  /** Constructor for class VMGJob.
   *
   * @param _JobId id of the job
   * @param _density_grid sampled electron charge density from short-range solutions
   * @param _particle_positions position per nuclei
   * @param _particle_charges charges per nuclei
   * @param _near_field_cells number of grid-points used to smear our nuclei charge
   */
  VMGJob(const JobId_t _JobId,
      const SamplingGrid &density_grid,
      const std::vector< std::vector< double > > &_particle_positions,
      const std::vector< double > &_particle_charges,
      const size_t _near_field_cells);
  virtual ~VMGJob();

  FragmentResult::ptr Work();

private:
  void InitVMG();

private:
  //!> sampled density required as input
  const SamplingGrid density_grid;
  //!> positions of all nuclei
  const std::vector< std::vector< double > > particle_positions;
  //!> charges of all nuclei
  const std::vector< double > particle_charges;
  //!> near field cells used in smearing out core charge density
  const size_t near_field_cells;

private:
  //!> temporary instance to hold return data
  VMGData returndata;

private:
  /** private default cstor for serialization only
   */
  VMGJob();

  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & boost::serialization::base_object<FragmentJob>(*this);
    ar & const_cast< SamplingGrid &>(density_grid);
    ar & const_cast< std::vector< std::vector< double > > &>(particle_positions);
    ar & const_cast< std::vector< double > &>(particle_charges);
    ar & const_cast< size_t &>(near_field_cells);
    ar & returndata;
  }
};

// we need to give this class a unique key for serialization
// its is only serialized through its base class FragmentJob
BOOST_CLASS_EXPORT_KEY(VMGJob)

#endif /* VMGJOB_HPP_ */
