/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * manipulateAtomsTest.cpp
 *
 *  Created on: Feb 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "manipulateAtomsTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>
#include <boost/bind.hpp>

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Actions/ManipulateAtomsProcess.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/ActionHistory.hpp"

#include "World.hpp"
#include "atom.hpp"

#include "DummyUI.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( manipulateAtomsTest );

// some stubs
class AtomStub : public atom {
public:
  AtomStub(int _id) :
  atom(),
  manipulated(false),
  id(_id)
  {}

  virtual atomId_t getId(){
    return id;
  }

  virtual void doSomething(){
    manipulated = true;
  }

  bool manipulated;
private:
  atomId_t id;
};

class countObserver : public Observer{
public:
  countObserver() :
    Observer("countObserver"),
    count(0)
    {}
  virtual ~countObserver(){}

  void update(Observable *){
    count++;
  }

  void subjectKilled(Observable *)
  {}

  int count;
};

// set up and tear down
void manipulateAtomsTest::setUp(){
  hasDescriptor = false;
  ActionHistory::init();
  World::getInstance();
  // TODO: find a way to really reset the factory to a clean state in tear-down
  if(!hasDescriptor){
    UIFactory::registerFactory(new DummyUIFactory::description());
    hasDescriptor = true;
  }
  UIFactory::makeUserInterface("Dummy");
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= new AtomStub(i);
    World::getInstance().registerAtom(atoms[i]);
  }
}
void manipulateAtomsTest::tearDown(){
  World::purgeInstance();
  ActionRegistry::purgeInstance();
  ActionHistory::purgeInstance();
  {
    UIFactory::purgeInstance();
    hasDescriptor = false;
  }
}

static void operation(atom* _atom){
  AtomStub *atom = dynamic_cast<AtomStub*>(_atom);
  CPPUNIT_ASSERT(atom);
  atom->doSomething();
}


void manipulateAtomsTest::testManipulateSimple(){
  ManipulateAtomsProcess *proc = World::getInstance().manipulateAtoms(boost::bind(operation,_1),"FOO",AllAtoms());
  proc->call();
  std::vector<atom*> allAtoms = World::getInstance().getAllAtoms(AllAtoms());
  std::vector<atom*>::iterator iter;
  for(iter=allAtoms.begin();iter!=allAtoms.end();++iter){
    AtomStub *atom;
    atom = dynamic_cast<AtomStub*>(*iter);
    CPPUNIT_ASSERT(atom);
    CPPUNIT_ASSERT(atom->manipulated);
  }
}

void manipulateAtomsTest::testManipulateExcluded(){

  ManipulateAtomsProcess *proc = World::getInstance().manipulateAtoms(boost::bind(operation,_1),"FOO",AllAtoms() && !AtomById(ATOM_COUNT/2));
  proc->call();
  std::vector<atom*> allAtoms = World::getInstance().getAllAtoms(AllAtoms());
  std::vector<atom*>::iterator iter;
  for(iter=allAtoms.begin();iter!=allAtoms.end();++iter){
    AtomStub *atom;
    atom = dynamic_cast<AtomStub*>(*iter);
    CPPUNIT_ASSERT(atom);
    if(atom->getId()!=(int)ATOM_COUNT/2)
      CPPUNIT_ASSERT(atom->manipulated);
    else
      CPPUNIT_ASSERT(!atom->manipulated);
  }
}

void manipulateAtomsTest::testObserver(){
  countObserver *obs = new countObserver();
  World::getInstance().signOn(obs);
  ManipulateAtomsProcess *proc = World::getInstance().manipulateAtoms(boost::bind(operation,_1),"FOO",AllAtoms());
  proc->call();

  CPPUNIT_ASSERT_EQUAL(1,obs->count);
  World::getInstance().signOff(obs);
  delete obs;
}
