/*
 * atom.hpp
 *
 *  Created on: Aug 3, 2009
 *      Author: heber
 */

#ifndef ATOM_HPP_
#define ATOM_HPP_

using namespace std;

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <list>
#include <vector>

#include "atom_atominfo.hpp"
#include "atom_bondedparticle.hpp"
#include "atom_graphnode.hpp"
#include "atom_particleinfo.hpp"
#include "Atom/TesselPoint.hpp"
#include "types.hpp"

#include "CodePatterns/enumeration.hpp"

/****************************************** forward declarations *****************************/

class AtomicInfo;
class Vector;
class World;
class molecule;
class Shape;

/********************************************** declarations *******************************/

/** Single atom.
 * Class incorporates position, type
 */
class atom : public GraphNode, public BondedParticle, public TesselPoint {
  friend atom* NewAtom(atomId_t);
  friend void  DeleteAtom(atom*);
public:
    atom *father;   //!< In many-body bond order fragmentations points to originating atom
    int *sort;      //!< sort criteria

  /** Clones this atom.
   *
   * Does not clone the bonds!
   *
   * @return reference to atom
   */
  virtual atom *clone();

  /** Pushes back another step in all trajectory vectors.
   *
   * This allows to extend all trajectories contained in different classes
   * consistently. This is implemented by the topmost class which calls the
   * real functions, \sa AppendTrajectoryStep(), by all necessary subclasses.
   */
  virtual void UpdateSteps();

  /** Output of a single atom with given numbering.
   * \param ElementNo cardinal number of the element
   * \param AtomNo cardinal number among these atoms of the same element
   * \param *out stream to output to
   * \param *comment commentary after '#' sign
   * \return true - \a *out present, false - \a *out is NULL
   */
  bool OutputIndexed(ofstream * const out, const int ElementNo, const int AtomNo, const char *comment = NULL) const;

  /** Output of a single atom with numbering from array according to atom::type.
   * \param *ElementNo cardinal number of the element
   * \param *AtomNo cardinal number among these atoms of the same element
   * \param *out stream to output to
   * \param *comment commentary after '#' sign
   * \return true - \a *out present, false - \a *out is NULL
   */
  bool OutputArrayIndexed(ostream * const out,const enumeration<const element*>&, int *AtomNo, const char *comment = NULL) const;

  /** Initialises the component number array.
   * Size is set to atom::ListOfBonds.size()+1 (last is th encode end by -1)
   */
  void InitComponentNr();

  /** Resets GraphNr to -1.
   *
   */
  void resetGraphNr();

  /** Check whether father is equal to given atom.
   * \param *ptr atom to compare father to
   * \param **res return value (only set if atom::father is equal to \a *ptr)
   */
  void EqualsFather ( const atom *ptr, const atom **res ) const;

  /** States whether the given \a *ptr is our father.
   *
   * @param ptr atom to compare atom::Father with
   * @return true - \a *ptr is father, false - not
   */
  bool isFather(const atom *ptr);

  /** If we are copy of copy, we are linked to be just a copy.
   *
   */
  void CorrectFather();

  /** Climbs up the father list until NULL, last is returned.
   * \return true father, i.e. whose father points to itself, NULL if it could not be found or has none (added hydrogen)
   */
  atom *GetTrueFather();

  /** Const version of \sa GetTrueFather().
   * \return true father, i.e. whose father points to itself, NULL if it could not be found or has none (added hydrogen)
   */
  const atom *GetTrueFather() const;

  /** Compares the indices of \a this atom with a given \a ptr.
   * \param ptr atom to compare index against
   * \return true - this one's is smaller, false - not
   */
  bool Compare(const atom &ptr) const;

  /** Returns distance to a given vector.
   * \param origin vector to calculate distance to
   * \return distance
   */
  double DistanceToVector(const Vector &origin) const;

  /** Returns squared distance to a given vector.
   * \param origin vector to calculate distance to
   * \return distance squared
   */
  double DistanceSquaredToVector(const Vector &origin) const;
  /** Checks whether atom is within the given box.
   * \param offset offset to box origin
   * \param *parallelepiped box matrix
   * \return true - is inside, false - is not
   */
  bool IsInShape(const Shape&) const;

  // getter and setter

  /**
   * returns the World that contains this atom.
   * Use this if you need to get the world without locking
   * the singleton for example.
   *
   */
  World *getWorld();
  void setWorld(World*);

  virtual atomId_t getId() const;
  virtual bool changeId(atomId_t newId);

  /**
   * this function sets the Id without notifying the world. Only use it, if the world has already
   * gotten an ID for this Atom.
   */
   virtual void setId(atomId_t);

   /** Returns pointer to the molecule which atom belongs to.
    * \return containing molecule
    */
   molecule* getMolecule() const;

   /** Erases the atom in atom::mol's list of atoms and sets it to zero.
    */
   void removeFromMolecule();

   /** Changes the molecule internal ParticleInfo::Nr of this atom.
    *
    * @param newNr new ParticleInfo::Nr to set
    * @return true - change successful, false - changed not successful, id remains the old one
    */
   bool changeNr(int newNr);

   /** Getter for ParticleInfo::Nr of the atom.
    *
    * @return index
    */
   int getNr() const;

   // Output operator
   std::ostream & operator << (std::ostream &ost) const;

  protected:

    /**
     * Protected constructor to ensure construction of atoms through the world.
     * see World::createAtom()
     */
    atom();

    /**
     * Protected copy-constructor to ensure construction of atoms by cloning.
     * see atom::clone()
     */
    atom(class atom *pointer);

    /**
     * Protected destructor to ensure destruction of atoms through the world.
     * see World::destroyAtom()
     */
    virtual ~atom();
  private:
    friend class molecule;
    friend class AtomicInfo;
    /** Makes the atom be contained in the new molecule \a *_mol.
     * Uses atom::removeFromMolecule() to delist from old molecule.
     * \param *_mol pointer to new molecule
     */
    void setMolecule(molecule*);

    /** Makes the atom be contained in the no molecule.
     * Use atom::removeFromMolecule() to delist from old molecule,
     * this assume that the molecule already knows about it.
     */
    void unsetMolecule();


  private:
    molecule *mol; // !< the molecule this atom belongs to
    World* world;
    atomId_t id;
};

/**
 * Global output operator for class atom.
 */
std::ostream & operator << (std::ostream &ost, const atom &_atom);

/**
 * internal method used by the world. Do not use if you don't know what you are doing.
 * You might get burned...
 * Use World::createAtom() instead.
 */
atom* NewAtom(atomId_t _id);

/**
* internal method used by the world. Do not use if you don't know what you are doing.
 * You might get burned...
 * Use World::destroyAtom() instead.
 */
void  DeleteAtom(atom*);

/**
 * Simple function to compare atoms by their elements to allow sorting of atoms by this criteria
 */
bool compareAtomElements(atom* atom1,atom* atom2);


#endif /* ATOM_HPP_ */
