/*
 * AtomSet.hpp
 *
 *  Created on: Jul 8, 2010
 *      Author: crueger
 */

#ifndef OBSERVEDCONTAINER_HPP_
#define OBSERVEDCONTAINER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iterator>
#include <utility>

#include "CodePatterns/Observer/ObservedIterator.hpp"
#include "CodePatterns/Observer/UnobservedIterator.hpp"

/**
 * Forward to std::map<atomId_t,atom*> that allows production of observed and
 * unobserved iterators to all atoms.
 *
 * Imported via typedef into class World. Defined here to avoid bloating of the
 * World class
 */
template <class Container, class Iterator = ObservedIterator<Container> >
class ObservedContainer
{
public:
  typedef Container set_t;
  // this iterator can be used for internal purposes...
  // no lock used here
  typedef typename set_t::iterator                              internal_iterator;
  typedef typename std::reverse_iterator<internal_iterator>     reverse_internal_iterator;

  typedef Iterator iterator;
  // typedefs for iterator structure
  typedef typename set_t::const_iterator                        const_iterator;
  typedef typename std::reverse_iterator<iterator>              reverse_iterator;
  typedef typename std::reverse_iterator<const_iterator>        const_reverse_iterator;

  // some more typedefs for STL-Structure
  typedef typename set_t::key_type               key_type;
  typedef typename set_t::mapped_type            mapped_type;
  typedef typename set_t::value_type             value_type;
  typedef typename set_t::key_compare            key_compare;
  typedef typename set_t::allocator_type         allocator_type;

  ObservedContainer(Observable*);
  ObservedContainer(const ObservedContainer&);
  virtual ~ObservedContainer();

  // all the functions from STL-map (forwards to content)
  ObservedContainer& operator=(const ObservedContainer&);

  iterator                      begin();
  const_iterator                begin()         const;
  iterator                      end();
  const_iterator                end()           const;
  reverse_iterator              rbegin();
  const_reverse_iterator        rbegin()        const;
  reverse_iterator              rend();
  const_reverse_iterator        rend()          const;

  bool empty() const;
  size_t size() const;
  size_t max_size() const;

  mapped_type &operator[](const key_type&);

  std::pair<iterator,bool> insert (const value_type&);

  size_t erase ( const key_type& x );
  void clear();

  iterator              find ( const key_type& x );
  const_iterator        find ( const key_type& x ) const;

  size_t count ( const key_type& x ) const;
  internal_iterator             begin_internal();
  internal_iterator             end_internal();
  reverse_internal_iterator     rbegin_internal();
  reverse_internal_iterator     rend_internal();

  set_t &getContent();
private:
  set_t content;
  Observable *obs;
};

#endif /* OBSERVEDCONTAINER_HPP_ */
