/*
 * AtomicInstance.hpp
 *
 *  Created on: Apr 26, 2016
 *      Author: heber
 */


#ifndef ATOMICINSTANCE_HPP_
#define ATOMICINSTANCE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/thread/mutex.hpp>

class AtomicInstanceTest;

/** The AtomicInstance class wraps another class with a mutex that is locked
 * on construction and unlocked on destruction. Hence, only a single entity
 * of this instance can be present at any single given time and thus allows
 * convenient atomic access to the state of the wrapped class.
 *
 * This is not perfect. The ref to the internal class may be stored elsewhere
 * and be used after the scope is left and the wrapping AtomicInstance is
 * destroyed.
 *
 * It is up to the user to maintain access to the class only within the lifetime
 * of this AtomicInstance object, i.e. to use it only within scopes.
 *
 * Note that this instance can be moved but not copied. That is on assignment
 * the contained pointer is NULL'ed in the right-hand side, regardless of whether
 * the rhs is const or not.
 */
template <class T>
class AtomicInstance
{
  //!> grant unit test access to private parts
  friend class AtomicInstanceTest;
public:
  /** Restrict copy cstor to only moves.
   *
   * \warning This resets \a content in \a rhs to mark it as being copied.
   *
   * \param rhs right-hand side, its \a content is NULL after this
   */
  AtomicInstance(AtomicInstance<T>& rhs);

  /** Restrict copy cstor to only moves.
   *
   * \warning This resets \a content in \a rhs to mark it as being copied.
   *
   * \param rhs right-hand side, its \a content is NULL after this
   */
  AtomicInstance(const AtomicInstance<T>& rhs);

  /** Cstor for wrapping the given instant \a _content.
   *
   * \param _content instance to wrap
   */
  explicit AtomicInstance(T* _content);

  /** Dstor to unlock the mutex.
   *
   */
  ~AtomicInstance();

  /** Getter for the ref to the internal object.
   *
   * \warning This must not be stored outside the scope of the AtomicInstance.
   *          We cannot prevent its copying.
   */
  T& operator*();

  /** Getter for the ref to the internal object.
   *
   * \warning This must not be stored outside the scope of the AtomicInstance.
   *          We cannot prevent its copying.
   */
  const T& operator*() const;

private:
  /** Disallow empty instance cstor.
   *
   */
  AtomicInstance();

  /** Restrict assignment operator.
   *
   * \param rhs right-hand side, its \a content is NULL after this
   */
  AtomicInstance<T>& operator=(const AtomicInstance<T>& rhs);

private:

  //!> internal ptr to the protected instance, not (un)locking when NULL
  mutable T* content;

  //!> a lock for atomic access to the state of the instance
  static boost::mutex atomicLock;
};

#include "AtomicInstance_impl.hpp"

#endif /* ATOMICINSTANCE_HPP_ */
