/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * PotentialFactory.cpp
 *
 *  Created on: 30.11.2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "PotentialFactory.hpp"

#include <boost/assign/list_of.hpp>
#include <boost/lambda/lambda.hpp>

#include "CodePatterns/Singleton_impl.hpp"

#include "Potentials/SerializablePotential.hpp"
#include "Potentials/Specifics/ManyBodyPotential_Tersoff.hpp"
#include "Potentials/Specifics/PairPotential_Angle.hpp"
#include "Potentials/Specifics/PairPotential_Harmonic.hpp"
#include "Potentials/Specifics/PairPotential_Morse.hpp"
#include "Potentials/Specifics/SaturationPotential.hpp"

// static defines
PotentialFactory::NameToType_t PotentialFactory::NameToType =
    boost::assign::map_list_of
    ("tersoff", tersoff)
    ("morse", morse)
    ("harmonic", harmonic)
    ("harmonic_angle", harmonic_angle)
    ("saturation", saturation)
  ;
PotentialFactory::TypeToName_t PotentialFactory::TypeToName =
    boost::assign::map_list_of
    (tersoff, "tersoff")
    (morse, "morse")
    (harmonic, "harmonic")
    (harmonic_angle, "harmonic_angle")
    (saturation, "saturation")
  ;


FunctionModel *PotentialFactory::createInstance(
    const std::string &potentialtype,
    const SerializablePotential::ParticleTypes_t &types) const
{
  ASSERT (NameToType.count(potentialtype),
      "PotentialFactory::createInstance() - cannot find potential of name "+potentialtype);
  switch (NameToType[potentialtype]) {
  case tersoff:
      return new ManyBodyPotential_Tersoff(types);
  case morse:
      return new PairPotential_Morse(types);
  case harmonic:
      return new PairPotential_Harmonic(types);
  case harmonic_angle:
      return new PairPotential_Angle(types);
  case saturation:
      return new SaturationPotential(types);
  default:
    ASSERT(0, "PotentialFactory::createInstance() - unknown potential desired to create.");
  }
  return NULL;
}

CONSTRUCT_SINGLETON(PotentialFactory)
