/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserPsi3UnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserPsi3UnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <boost/any.hpp>

#include "World.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "CodePatterns/Assert.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/Psi3Parser.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserPsi3UnitTest );

static string hydrogenPsi3_RHF ="% Created by MoleCuilder\n\
psi: (\n\
\tlabel = \"cc-pVTZ SCF H2\"\n\
\tjobtype = sp\n\
\twfn = scf\n\
\tmaxiter = 80\n\
\treference = rhf\n\
\tbasis = \"cc-pVTZ\"\n\
\tfreeze_core = no\n\
\tunits = bohr\n\
\tgeometry = (\n\
\t\t( H     0.0000      0.0000  0.0 )\n\
\t\t( H     0.0000      0.0000  0.74 )\n\
\t)\n\
\torigin = (0.0 0.0 0.0)\n\
)\n"; // tested with ???

static string hydrogenPsi3_ROHF ="% Created by MoleCuilder\n\
psi: (\n\
\tlabel = \"cc-pVTZ SCF H2\"\n\
\tjobtype = sp\n\
\twfn = scf\n\
\tmaxiter = 80\n\
\treference = rohf\n\
\tbasis = \"cc-pVTZ\"\n\
\tfreeze_core = no\n\
\tunits = bohr\n\
\tgeometry = (\n\
\t\t( H     0.0000      0.0000  0.0 )\n\
\t\t( H     0.0000      0.0000  0.74 )\n\
\t)\n\
\torigin = (0.0 0.0 0.0)\n\
)\n"; // tested with ???

static string hydrogenPsi3_UHF ="% Created by MoleCuilder\n\
psi: (\n\
\tlabel = \"cc-pVTZ SCF H2\"\n\
\tjobtype = sp\n\
\twfn = scf\n\
\tmaxiter = 80\n\
\treference = uhf\n\
\tbasis = \"cc-pVTZ\"\n\
\tmultp = 2\n\
\t%charge = 2\n\
\tfreeze_core = no\n\
\tunits = bohr\n\
\tgeometry = (\n\
\t\t( H     0.0000      0.0000  0.0 )\n\
\t\t( H     0.0000      0.0000  0.74 )\n\
\t)\n\
\torigin = (0.0 0.0 0.0)\n\
)\n"; // tested with ???

static string hydrogenPsi3_TWOCON ="% Created by MoleCuilder\n\
psi: (\n\
\tlabel = \"cc-pVTZ SCF H2\"\n\
\tjobtype = sp\n\
\twfn = scf\n\
\tmaxiter = 80\n\
\treference = TWOCON\n\
\tbasis = \"cc-pVTZ\"\n\
\tmultp = 1\n\
\t%charge = 2\n\
\tsocc = ( 1 1 0 0 0 0 0 0)\n\
\tdocc = (0     0     0     0     0     0     0     0)\n\
\tfreeze_core = no\n\
\tunits = bohr\n\
\tgeometry = (\n\
\t\t( H     0.0000      0.0000  0.0 )\n\
\t\t( H     0.0000      0.0000  0.74 )\n\
\t)\n\
\torigin = (0.0 0.0 0.0)\n\
)\n"; // tested with ???

void ParserPsi3UnitTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  parser = new FormatParser<psi3>();

  params = &parser->getParams();

  World::getInstance();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
}

void ParserPsi3UnitTest::tearDown()
{
  params = NULL;
  delete parser;
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserPsi3UnitTest::ParameterDefaultTest() {
  // check default values
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::labelParam) == std::string("unknown job"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::SP]);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::wavefunctionParam) == params->ValidWavefunction[Psi3Parser_Parameters::SCF]);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::maxiterParam) == std::string("80"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::referenceParam) == params->ValidReference[Psi3Parser_Parameters::RHF]);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::basisParam) == std::string("cc-pVTZ"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::freeze_coreParam) == params->ValidFreezeCore[Psi3Parser_Parameters::YES]);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::unitsParam) == params->ValidUnits[Psi3Parser_Parameters::angstrom]);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::dertypeParam) == params->ValidDerivativeType[Psi3Parser_Parameters::NONE]);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::originParam) == std::string("(0.0\t0.0\t0.0)"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::multipParam) == std::string("1"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::chargeParam) == std::string("0"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::soccParam) == std::string("()"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::doccParam) == std::string("()"));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::subgroupParam) == std::string(""));
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::unique_axisParam) == params->ValidUniqueAxis[Psi3Parser_Parameters::X]);
}

void ParserPsi3UnitTest::ParameterCloneTest() {
  FormatParser_Parameters *clone = params->clone();
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::SP]);
  std::stringstream setvalue(params->ParamNames[Psi3Parser_Parameters::jobtypeParam]+"="+params->ValidJobtypes[Psi3Parser_Parameters::OPT]);
  setvalue >> *params;
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::OPT]);
  params->makeClone(*clone);
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::SP]);
}

void ParserPsi3UnitTest::ParameterSetterTest() {
  // test a jobtype
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::jobtypeParam]
        +" = "+params->ValidJobtypes[Psi3Parser_Parameters::OPT]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::OPT]);
  }
  // test a wavefunction
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::wavefunctionParam]
        +" = "+params->ValidWavefunction[Psi3Parser_Parameters::MP2]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::wavefunctionParam) == params->ValidWavefunction[Psi3Parser_Parameters::MP2]);
  }
  // test a reference
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::referenceParam]
        +" = "+params->ValidReference[Psi3Parser_Parameters::ROHF]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::referenceParam) == params->ValidReference[Psi3Parser_Parameters::ROHF]);
  }
  // test a unique_axis
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::unique_axisParam]
        +" = "+params->ValidUniqueAxis[Psi3Parser_Parameters::Y]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::unique_axisParam) == params->ValidUniqueAxis[Psi3Parser_Parameters::Y]);
  }
  // test a units
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::unitsParam]
        +" = "+params->ValidUnits[Psi3Parser_Parameters::bohr]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::unitsParam) == params->ValidUnits[Psi3Parser_Parameters::bohr]);
  }
  // test a dertype
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::dertypeParam]
        +" = "+params->ValidDerivativeType[Psi3Parser_Parameters::NONE]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::dertypeParam) == params->ValidDerivativeType[Psi3Parser_Parameters::NONE]);
  }
  // test a freeze_core
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::freeze_coreParam]
        +" = "+params->ValidFreezeCore[Psi3Parser_Parameters::LARGE]
    );
    setvalue >> *params;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::freeze_coreParam) == params->ValidFreezeCore[Psi3Parser_Parameters::LARGE]);
  }
  // test int
  {
    std::stringstream setvalue(
        params->ParamNames[Psi3Parser_Parameters::maxiterParam]
        +" = 500"
    );
    setvalue >> *params;
//    std::cout << "maxiter is "
//        << params->getString(Psi3Parser_Parameters::maxiterParam) << std::endl;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::maxiterParam) == std::string("500"));
  }
  // test whether unknown key fails
  std::cout << "The following Assertion warning is desired and does not indicate a failure of the test." << std::endl;
  {
    std::stringstream setvalue("sd = no");
#ifndef NDEBUG
    ASSERT_DO(Assert::Throw);
    CPPUNIT_ASSERT_THROW(setvalue >> *params, Assert::AssertionFailure);
#else
    setvalue >> *params;
#endif
//    std::cout << "Hessian is still "
//        << params->getString(Psi3Parser_Parameters::hessianParam) << std::endl;
    CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::OPT]);
  }
}

void ParserPsi3UnitTest::readPsi3Test() {
  stringstream input(hydrogenPsi3_RHF);
  // set some other parameter for jobtype
  params->setParameter(
      Psi3Parser_Parameters::jobtypeParam,
      params->ValidJobtypes[Psi3Parser_Parameters::OPT]
          );
  parser->load(&input);

  // check for jobtype from file
  CPPUNIT_ASSERT(params->getParameter(Psi3Parser_Parameters::jobtypeParam) == params->ValidJobtypes[Psi3Parser_Parameters::SP]);
  // check for 2 hydrogens
  CPPUNIT_ASSERT_EQUAL(2, World::getInstance().numAtoms());
  // check that positions are right
  Vector PositionSum;
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::const_iterator iter = atoms.begin();
      iter != atoms.end();
      ++iter)
    PositionSum += (*iter)->getPosition();
  CPPUNIT_ASSERT_EQUAL( PositionSum, Vector(0,0,0.74) );
}

void ParserPsi3UnitTest::writePsi3Test() {
//  // build up water molecule
//  string first;
//  string second;
//  atom *Walker = NULL;
//  Walker = World::getInstance().createAtom();
//  Walker->setType(8);
//  Walker->setPosition(Vector(0,0,0));
//  Walker = World::getInstance().createAtom();
//  Walker->setType(1);
//  Walker->setPosition(Vector(0.758602,0,0.504284));
//  Walker = World::getInstance().createAtom();
//  Walker->setType(1);
//  Walker->setPosition(Vector(0.758602,0,-0.504284));
//  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
//
//  // create two stringstreams, one stored, one created
//
//  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
//  {
//    // compare both configs for CLHF
//    stringstream output;
//    params->setParameter(
//        Psi3Parser_Parameters::theoryParam,
//        params->getTheoryName(Psi3Parser_Parameters::CLHF)
//            );
//    parser->save(&output, atoms);
//    stringstream input(waterPsi3_CLHF);
//    for (; std::getline(input, first) && std::getline(output, second); ) {
//      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
//      CPPUNIT_ASSERT(first == second);
//    }
//  }
//  {
//    // compare both configs for CLKS
//    stringstream output;
//    params->setParameter(
//        Psi3Parser_Parameters::theoryParam,
//        params->getTheoryName(Psi3Parser_Parameters::CLKS)
//            );
//    stringstream input(waterPsi3_CLKS);
//    for (; std::getline(input, first) && std::getline(output, second); ) {
//      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
//      CPPUNIT_ASSERT(first == second);
//    }
//  }
//  {
//    // compare both configs for MBPT2
//    stringstream output;
//    params->setParameter(
//        Psi3Parser_Parameters::theoryParam,
//        params->getTheoryName(Psi3Parser_Parameters::MBPT2)
//            );
//    stringstream input(waterPsi3_MBPT2);
//    for (; std::getline(input, first) && std::getline(output, second); ) {
//      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
//      CPPUNIT_ASSERT(first == second);
//    }
//  }
//  {
//    // compare both configs for MBPT2_R12
//    stringstream output;
//    params->setParameter(
//        Psi3Parser_Parameters::theoryParam,
//        params->getTheoryName(Psi3Parser_Parameters::MBPT2_R12)
//            );
//    stringstream input(waterPsi3_MBPT2_R12);
//    for (; std::getline(input, first) && std::getline(output, second); ) {
//      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
//      CPPUNIT_ASSERT(first == second);
//    }
//  }
}
