/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Psi3Parser.cpp
 *
 *  Created on: Oct 04, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <boost/tokenizer.hpp>
#include <string>

#include "CodePatterns/MemDebug.hpp"

#include "Psi3Parser.hpp"
#include "Psi3Parser_Parameters.hpp"

#include "atom.hpp"
#include "config.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/toString.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "periodentafel.hpp"
#include "World.hpp"

// declare specialized static variables
const std::string FormatParserTrait<psi3>::name = "psi3";
const std::string FormatParserTrait<psi3>::suffix = "in";
const ParserTypes FormatParserTrait<psi3>::type = psi3;

// a converter we often need
ConvertTo<bool> FormatParser<psi3>::Converter;

/** Constructor of Psi3Parser.
 *
 */
FormatParser< psi3 >::FormatParser() :
  FormatParser_common(new Psi3Parser_Parameters())
{}

/** Destructor of Psi3Parser.
 *
 */
FormatParser< psi3 >::~FormatParser()
{}

/** Load an MPQC config file into the World.
 * \param *file input stream
 */
void FormatParser< psi3 >::load(istream *file)
{
  bool Psi3Section = false;
  bool GeometrySection = false;
  char line[MAXSTRINGSIZE];
  typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
  boost::char_separator<char> sep("()");
  boost::char_separator<char> angularsep("<>");
  boost::char_separator<char> equalitysep(" =");
  boost::char_separator<char> whitesep(" \t");
  ConvertTo<double> toDouble;

  molecule *newmol = World::getInstance().createMolecule();
  newmol->ActiveFlag = true;
  // TODO: Remove the insertion into molecule when saving does not depend on them anymore. Also, remove molecule.hpp include
  World::getInstance().getMolecules()->insert(newmol);
  while (file->good()) {
    file->getline(line, MAXSTRINGSIZE-1);
    std::string linestring(line);
    LOG(3, "INFO: Current line is: " << line);
    if ((linestring.find(")") != string::npos) && (linestring.find("(") == string::npos)) {
      LOG(3, "INFO: Line contains ')' and no '(' (end of section): " << line);
      // ends a section which do not overlap
      if (GeometrySection)
        GeometrySection = false;
      else
        Psi3Section = false;
    }
    if (GeometrySection) { // we have an atom
      tokenizer tokens(linestring, sep);
//      if (tokens.size() != 2)
//        throw Psi3ParseException;
      tokenizer::iterator tok_iter = tokens.begin();
      ASSERT(tok_iter != tokens.end(),
          "FormatParser< psi3 >::load() - missing token for MoleculeSection in line "+linestring+"!");
      std::stringstream whitespacefilter(*++tok_iter);
      std::string element;
      whitespacefilter >> ws >> element;
      LOG(2, "INFO: element of atom is " << element);
      ASSERT(tok_iter != tokens.end(),
          "FormatParser< psi3 >::load() - missing token for MoleculeSection in line "+linestring+"!");
      std::string vector = *tok_iter;
      tokenizer vectorcomponents(vector, whitesep);
      Vector X;
//      if (vectorcomponents.size() != NDIM)
//        throw Psi3ParseException;
      tok_iter = vectorcomponents.begin();
      ++tok_iter;
      for (int i=0; i<NDIM; ++i) {
        LOG(4, "INFO: Current value is " << *tok_iter << ".");
        X[i] = toDouble(*tok_iter++);
      }
      LOG(2, "INFO: position of atom is " << X);
      // create atom
      atom *newAtom = World::getInstance().createAtom();
      newAtom->setType(World::getInstance().getPeriode()->FindElement(element));
      newAtom->setPosition(X);
      newmol->AddAtom(newAtom);
      DoLog(1) && (Log() << Verbose(1) << "Adding atom " << *newAtom << std::endl);
    }
    if ((Psi3Section) && (!GeometrySection)) {
      if (linestring.find("=") != string::npos) { // get param value
        tokenizer tokens(linestring, equalitysep);
        tokenizer::iterator tok_iter = tokens.begin();
        ASSERT(tok_iter != tokens.end(),
            "FormatParser< psi3 >::load() - missing token before '=' for Psi3Section in line "+linestring+"!");
        std::stringstream whitespacefilter(*tok_iter);
        std::string key;
        whitespacefilter >> ws >> key;
        //LOG(2, "INFO: key to check is: " << key);
        if (getParams().haveParameter(key)) {
          //LOG(2, "INFO: Line submitted to parameter is: " << linestring);
          std::stringstream linestream(linestring);
          linestream >> getParams();
        } else { // unknown keys are simply ignored as long as parser is incomplete
          LOG(3, "INFO: '"+key+"' is unknown and ignored.");
        }
      }
    }
    if ((linestring.find("geometry") != string::npos) && (linestring.find("(") != string::npos)) {
      LOG(3, "INFO: Line contains geometry and '(': " << line);
      GeometrySection = true;
    }
    if ((linestring.find("psi:") != string::npos) && (linestring.find("(") != string::npos)) {
      LOG(3, "INFO: Line contains psi: and '(': " << line);
      Psi3Section = true;
    }
  }
  // refresh atom::nr and atom::name
  newmol->getAtomCount();
}

/** Saves all atoms and data into a MPQC config file.
 * \param *file output stream
 * \param atoms atoms to store
 */
void FormatParser< psi3 >::save(ostream *file, const std::vector<atom *> &atoms)
{
//  Vector center;
////  vector<atom *> allatoms = World::getInstance().getAllAtoms();
//
//  // calculate center
//  for (std::vector<atom *>::const_iterator runner = atoms.begin();runner != atoms.end(); ++runner)
//    center += (*runner)->getPosition();
//  center.Scale(1./(double)atoms.size());
//
//  // first without hessian
//  if (file->fail()) {
//    DoeLog(1) && (eLog()<< Verbose(1) << "Cannot open psi3 output file." << endl);
//  } else {
//    *file << "% Created by MoleCuilder" << endl;
//    *file << "psi3: (" << endl;
//    *file << "\tsavestate = " << getParams().getParameter(Psi3Parser_Parameters::savestateParam) << endl;
//    *file << "\tdo_gradient = " << getParams().getParameter(Psi3Parser_Parameters::do_gradientParam) << endl;
//    if (Converter(getParams().getParameter(Psi3Parser_Parameters::hessianParam))) {
//      *file << "\tfreq<MolecularFrequencies>: (" << endl;
//      *file << "\t\tmolecule=$:molecule" << endl;
//      *file << "\t)" << endl;
//    }
//    const std::string theory = getParams().getParameter(Psi3Parser_Parameters::theoryParam);
//    if (theory == getParams().getTheoryName(Psi3Parser_Parameters::CLHF)) {
//      *file << "\tmole<" << getParams().getParameter(Psi3Parser_Parameters::theoryParam) << ">: (" << endl;
//      *file << "\t\tmolecule = $:molecule" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::basisParam) << " = $:basis" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::maxiterParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::maxiterParam)<< endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::memoryParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::memoryParam) << endl;
//      *file << "\t)" << endl;
//    } else if (theory == getParams().getTheoryName(Psi3Parser_Parameters::CLKS)) {
//      *file << "\tmole<" << getParams().getParameter(Psi3Parser_Parameters::theoryParam) << ">: (" << endl;
//      *file << "\t\tfunctional<StdDenFunctional>:(name=B3LYP)" << endl;
//      *file << "\t\tmolecule = $:molecule" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::basisParam) << " = $:basis" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::maxiterParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::maxiterParam)<< endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::memoryParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::memoryParam) << endl;
//      *file << "\t)" << endl;
//    } else if (theory == getParams().getTheoryName(Psi3Parser_Parameters::MBPT2)) {
//      *file << "\tmole<" << getParams().getParameter(Psi3Parser_Parameters::theoryParam) << ">: (" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::basisParam) << " = $:basis" << endl;
//      *file << "\t\tmolecule = $:molecule" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::memoryParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::memoryParam) << endl;
//      *file << "\t\treference<CLHF>: (" << endl;
//      *file << "\t\t\t" << getParams().getParameterName(Psi3Parser_Parameters::maxiterParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::maxiterParam)<< endl;
//      *file << "\t\t\t" << getParams().getParameterName(Psi3Parser_Parameters::basisParam) << " = $:basis" << endl;
//      *file << "\t\t\tmolecule = $:molecule" << endl;
//      *file << "\t\t\t" << getParams().getParameterName(Psi3Parser_Parameters::memoryParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::memoryParam) << endl;
//      *file << "\t\t)" << endl;
//      *file << "\t)" << endl;
//    } else if (theory == getParams().getTheoryName(Psi3Parser_Parameters::MBPT2_R12)) {
//      *file << "\tmole<" << getParams().getParameter(Psi3Parser_Parameters::theoryParam) << ">: (" << endl;
//      *file << "\t\tmolecule = $:molecule" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::basisParam) << " = $:basis" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::aux_basisParam) << " = $:abasis" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::stdapproxParam)
//          << " = \"" << getParams().getParameter(Psi3Parser_Parameters::stdapproxParam) << "\"" << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::nfzcParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::nfzcParam) << endl;
//      *file << "\t\t" << getParams().getParameterName(Psi3Parser_Parameters::memoryParam)
//          << " = " << getParams().getParameter(Psi3Parser_Parameters::memoryParam) << endl;
//      *file << "\t\tintegrals<IntegralCints>:()" << endl;
//      *file << "\t\treference<CLHF>: (" << endl;
//      *file << "\t\t\tmolecule = $:molecule" << endl;
//      *file << "\t\t\t" << getParams().getParameterName(Psi3Parser_Parameters::basisParam) << " = $:basis" << endl;
//      *file << "\t\t\tmaxiter = " << getParams().getParameter(Psi3Parser_Parameters::maxiterParam) << endl;
//      *file << "\t\t\tmemory = " << getParams().getParameter(Psi3Parser_Parameters::memoryParam) << endl;
//      *file << "\t\t\tintegrals<" << getParams().getParameter(Psi3Parser_Parameters::integrationParam) << ">:()" << endl;
//      *file << "\t\t)" << endl;
//      *file << "\t)" << endl;
//    } else {
//      DoeLog(0) && (eLog() << Verbose(0)
//          << "Unknown level of theory requested for MPQC output file." << std::endl);
//    }
//    *file << ")" << endl;
//    *file << "molecule<Molecule>: (" << endl;
//    *file << "\tunit = " << (World::getInstance().getConfig()->GetIsAngstroem() ? "angstrom" : "bohr" ) << endl;
//    *file << "\t{ atoms geometry } = {" << endl;
//    // output of atoms
//    for (std::vector<atom *>::const_iterator AtomRunner = atoms.begin(); AtomRunner != atoms.end(); ++AtomRunner) {
//      (*AtomRunner)->OutputMPQCLine(file, &center);
//    }
//    *file << "\t}" << endl;
//    *file << ")" << endl;
//    *file << "basis<GaussianBasisSet>: (" << endl;
//    *file << "\tname = \"" << getParams().getParameter(Psi3Parser_Parameters::basisParam) << "\"" << endl;
//    *file << "\tmolecule = $:molecule" << endl;
//    *file << ")" << endl;
//    if (theory == getParams().getTheoryName(Psi3Parser_Parameters::MBPT2_R12)) {
//      *file << "% auxiliary basis set specification" << endl;
//      *file << "\tabasis<GaussianBasisSet>: (" << endl;
//      *file << "\tname = \"" << getParams().getParameter(Psi3Parser_Parameters::aux_basisParam) << "\"" << endl;
//      *file << "\tmolecule = $:molecule" << endl;
//      *file << ")" << endl;
//    }
//  }
}


