/*
 * QtMoleculeList.hpp
 *
 *  Created on: Jan 21, 2010
 *      Author: crueger
 */

#ifndef QTMOLECULELIST_HPP_
#define QTMOLECULELIST_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui/QItemSelection>
#include <QtGui/QStandardItem>
#include <QtGui/QStandardItemModel>
#include <QTimer>

#include <boost/bimap.hpp>
#include <boost/thread/recursive_mutex.hpp>
#include <map>
#include <set>
#include <string>

#include "CodePatterns/Observer/Observer.hpp"

#include "UIElements/Views/Qt4/MoleculeList/QtMoleculeItem.hpp"

#include "types.hpp"

class molecule;
class MoleculeListClass;
class QtMoleculeItem;
class QtMoleculeListView;

class QtMoleculeList : public QStandardItemModel, public Observer
{
  Q_OBJECT

public:
  QtMoleculeList();
  virtual ~QtMoleculeList();

protected:
  virtual void update(Observable *publisher);
  virtual void recieveNotification(Observable *publisher, Notification_ptr notification);
  virtual void subjectKilled(Observable *publisher);

  void refill();

  QVariant headerData(int section, Qt::Orientation orientation, int role) const;

private slots:
  void moleculeChanged();
//  void visibilityChanged(QStandardItem*, int);
  void checkForVisibilityChange(QStandardItem*);
  void checkState();

  friend class QtMoleculeItem;

  std::string readdItem(QtMoleculeItem *_molitem);

signals:
  void moleculeSelected(molecule*);
  void moleculeUnSelected(molecule*);
  void moleculesVisibilityChanged(const moleculeId_t, bool);

private:

  bool ChangingChildrensVisibility;

  mutable boost::recursive_mutex listAccessing_mutex;
  mutable boost::recursive_mutex refill_mutex;
  mutable boost::recursive_mutex map_mutex;

  //!> informs the view in regular intervals about updates
  QTimer *update_timer;

  //!> how often per second update is signalled to view
  static const unsigned int update_times_per_second;

  friend class QtMoleculeListView;

  bool isMoleculeItemPresent(const moleculeId_t _molid) const;
  QtMoleculeItem * MoleculeIdToItem(const moleculeId_t _molid) const;
  const moleculeId_t ItemToMoleculeId(const QtMoleculeItem * const _item) const;
  const QModelIndex MoleculeIdToIndex(const moleculeId_t _id) const;
  const moleculeId_t IndexToMoleculeId(const QModelIndex &_index) const;
  QtMoleculeItem * getSpecificMoleculeItem(
      const QtMoleculeItem * const _item,
      const enum QtMoleculeItem::COLUMNTYPES _type) const;
  bool isGroupItemPresent(const std::string &_formula) const;
  const std::string& GroupItemToFormula(const QStandardItem * const _item) const;
  QStandardItem * FormulaToGroupItem(const std::string &_formula) const;
  QStandardItem * getSpecificGroupItem(
      const QStandardItem * const _item,
      const enum QtMoleculeItem::COLUMNTYPES _type) const;

  std::string addMolecule(const molecule * const _mol);
  void addGroupItem(QStandardItem *&mainitem, const std::string &_molecule_formula);
  void addMoleculeItem(QStandardItem *_groupitem, const moleculeId_t _molid, const std::string &molecule_formula);
  void removeMoleculeItem(QtMoleculeItem * const _item);
  int setOccurrence(QStandardItem * const _groupitem);
  bool areAnyItemsDirty();
  void setVisibilityForMoleculeItem(QtMoleculeItem* _item);
  void setVisibilityForGroupItem(QStandardItem* _item);

  void informDirtyState(
      const moleculeId_t _id,
      const QtMoleculeItem::COLUMNTYPES _type,
      const QtMoleculeItem::MoveTypes _movetype);

  void receiveSubjectKilled(const moleculeId_t _id);

  void updateItemStates();

  typedef std::map<std::string, unsigned int> FormulaVisibilityCountMap_t;
  FormulaVisibilityCountMap_t FormulaVisibilityCountMap;

  typedef boost::bimap<std::string, QStandardItem*> FormulaTreeItemBiMap_t;
  //!> map of (unique) formulas in the world
  FormulaTreeItemBiMap_t FormulaItemBiMap;

  typedef std::map<moleculeId_t, std::string> MoleculeFormulaMap_t;
  //!> map of (unique) formulas in the world
  MoleculeFormulaMap_t MoleculeFormulaMap;

  typedef boost::bimap<moleculeId_t, QtMoleculeItem*> MoleculeItemBiMap_t;
  MoleculeItemBiMap_t MoleculeItemBiMap;

  //!> callback function to hand over to items that inform about requiring update
  const QtMoleculeItem::emitDirtyState_t callback_DirtyItems;

  //!> callback function to hand over to items that inform about called subjectKilled()
  const QtMoleculeItem::emitSubjectKilledState_t callback_subjectKilledItems;

  typedef std::map<moleculeId_t, unsigned int> KilledItemsPerMolecule_t;
  //!> takes note of how many items have already been killed for a specific row/molecule
  KilledItemsPerMolecule_t KilledItemsPerMolecule;

  typedef std::set< moleculeId_t > list_of_molecules_t;
  typedef std::set< std::pair<moleculeId_t, QtMoleculeItem::COLUMNTYPES> > list_of_molecule_items_t;
  typedef std::set< std::pair<std::string, QtMoleculeItem::COLUMNTYPES> > list_of_group_items_t;
  //!> list of molecule items that need an update
  list_of_molecule_items_t dirtyMolItems;
  //!> list of molecule items that need an update of visibility
  list_of_molecules_t visibilityMolItems;
  //!> list of group items that need an update
  list_of_group_items_t dirtyGroupItems;
  //!> list of group items that need an update of visibility
  list_of_group_items_t visibilityGroupItems;
  //!> list of new molecules which need to be added
  std::vector<moleculeId_t> newMolecules;
  //!> list of molecules that have been removed
  std::vector<moleculeId_t> removedMolecules;
  //!> list of molecule items to molecule's whose formulas has changed and need to be moved
  list_of_molecules_t toBeMovedItems;
};

#endif /* QTMOLECULELIST_HPP_ */
