/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Listener.cpp
 *
 *  Created on: 22.02.2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "Listener.hpp"

#include "CodePatterns/Log.hpp"

/** Constructor for class Listener.
 *
 * @param io_service io service to obtain connections from
 * @param port service or port number to listen on
 */
Listener::Listener(boost::asio::io_service& io_service, unsigned short port) :
    Exitflag(OkFlag),
    acceptor_(io_service)
{
//  // enable abort reporting
//  boost::asio::socket_base::enable_connection_aborted option(true);
//  acceptor_.set_option(option);
  // state endpoint
  boost::asio::ip::tcp::endpoint endpoint(boost::asio::ip::tcp::v4(), port);
  acceptor_.open(endpoint.protocol());
  acceptor_.set_option(boost::asio::ip::tcp::acceptor::reuse_address(true));
  boost::system::error_code ec;
  acceptor_.bind(endpoint, ec);
  if (!ec) {
    acceptor_.listen();
    LOG(1, "Listening on port " << port << ".");
  } else {
    ELOG(0, "Binding failed on port " << port << ": " << ec.message());
  }
}

/** Destructor for class Listener.
 *
 */
Listener::~Listener()
{}

/** Initialises the socket in a \a Connection for listening.
 *
 */
void Listener::initiateSocket()
{
  // Start an accept operation for worker connections.
  connection.reset(new Connection(acceptor_.get_io_service()));
  acceptor_.async_accept(connection->socket(),
    boost::bind(&Listener::handle_Accept, this,
      boost::asio::placeholders::error, connection));
}

/** Initialises the socket in a \a Connection for listening.
 *
 */
void Listener::closeSocket()
{
  boost::system::error_code errorcode;

  if (connection->socket().is_open())
  {
    LOG(1, "INFO: Shutting socket down ...");
    // Boost documentation recommends calling shutdown first
    // for "graceful" closing of socket.
    connection->socket().shutdown(boost::asio::ip::tcp::socket::shutdown_both, errorcode);
    if (errorcode)
    {
      Exitflag = ErrorFlag;
      ELOG(1, "socket.shutdown error: " << errorcode.message());
    }

    LOG(1, "INFO: Closing socket ...");
    connection->socket().close(errorcode);
    if (errorcode)
    {
      Exitflag = ErrorFlag;
      ELOG(1, "socket.close error: " << errorcode.message());
    }
  } else {
    LOG(1, "INFO: Socket is not open, no need to shut down.");
  }
}

