/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MPQCCommandJob.cpp
 *
 *  Created on: Feb 05, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "MPQCCommandJob.hpp"

#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/tokenizer.hpp>
#include <sstream>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "LinearAlgebra/defs.hpp"

const std::string MPQCCommandJob::keyword_hartreefock_energy("total scf energy");
const std::string MPQCCommandJob::keyword_moellerplesset_energy("MP2 energy [au]:");
const std::string MPQCCommandJob::keyword_hartreefock_forces("Total Gradient");
const std::string MPQCCommandJob::keyword_moellerplesset_forces("Total MP2 gradient [au]:");

/** Default constructor for class MPQCCommandJob.
 *
 */
MPQCCommandJob::MPQCCommandJob()
{}

/** Constructor for class MPQCCommandJob.
 *
 * @param _inputfile string of inputfile contents
 * @param _JobId id of this job
 * @param command mpqc command, "mpqc" as default
 */
MPQCCommandJob::MPQCCommandJob(
    const std::string &_inputfile,
    const JobId_t _JobId,
    const std::string &command) :
    SystemCommandJob(command, _inputfile, _JobId)
{}

/** Destructor for class MPQCCommandJob.
 *
 */
MPQCCommandJob::~MPQCCommandJob()
{}

/** Extracts energy and forces from the output of the command.
 *
 * @param resultstring output of the command to parse
 * @return FragmentResult with energy and forces
 */
FragmentResult::ptr MPQCCommandJob::extractResult(const std::string &resultstring)
{
  std::stringstream returnstream;
  // tokenizers
  typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
  boost::char_separator<char> keyvalue_separator("=:");
  boost::char_separator<char> whitespace(" \t");

  // split into lines
  std::vector<std::string> lines;
  boost::split(lines, resultstring, boost::is_any_of("\n"));

  // go through each line
  bool gradient_section = false;
  for (std::vector<std::string>::const_iterator iter = lines.begin();
      iter != lines.end();++iter) {
    LOG(3, "DEBUG: Current line is '" << *iter << "'.");
    if (gradient_section) {
      tokenizer tokens(*iter, whitespace);
      if (*iter != std::string("")) {
        tokenizer::iterator tok_iter = tokens.begin();
        tok_iter++;
        tok_iter++;
        MPQCData::vector_type forcevector(NDIM);
        try { // first col is index, second is element
          for (size_t index = 0;index < NDIM; ++index)
            forcevector[index] = boost::lexical_cast<double>(*(tok_iter++));
          LOG(2, "INFO: Recognized force vector in '"+*iter+"'.");
          data.forces.push_back( forcevector );
        } catch(boost::bad_lexical_cast){
          LOG(2, "INFO: Did not recognize a force vector, hence ending section at '"+*iter+"'.");
          gradient_section = false;
        }
      } else {
        LOG(2, "INFO: Recognized gradient section end in '"+*iter+"'.");
        gradient_section = false;
      }
    }
    // extract energy ("total scf energy") ...
    if (((*iter).find(keyword_hartreefock_energy) != std::string::npos)
        || ((*iter).find(keyword_moellerplesset_energy) != std::string::npos))
    {
      LOG(2, "INFO: Recognized energy in '"+*iter+"'.");
      tokenizer tokens(*iter, keyvalue_separator);
      tokenizer::iterator tok_iter = tokens.begin();
      tok_iter++;
      std::stringstream energystring(*tok_iter);
      energystring >> data.energy;
    }
    // ... and forces ("Total Gradient") from resultstring
    if (((*iter).find(keyword_hartreefock_forces) != std::string::npos)
        || ((*iter).find(keyword_moellerplesset_forces) != std::string::npos))
    {
      LOG(2, "INFO: Recognized gradient section init in '"+*iter+"'.");
      gradient_section=true;
      data.forces.clear();
    }
  }

  // place into returnstream
  boost::archive::text_oarchive oa(returnstream);
  oa << data;

  FragmentResult::ptr s( new FragmentResult(getId(), returnstream.str()) );
  return s;
}

bool MPQCCommandJob::operator==(const MPQCCommandJob &other) const
{
  if (data != other.data) {
    LOG(2, "INFO: data's of both MPQCCommandJob don't match.");
    return false;
  }
  return (static_cast<const SystemCommandJob &>(*this)
      == static_cast<const SystemCommandJob &>(other));
}


// we need to explicitly instantiate the serialization functions as
// its is only serialized through its base class FragmentJob
BOOST_CLASS_EXPORT_IMPLEMENT(MPQCCommandJob)
